//
//  Copyright (c) 2005 Joël Amblard - joel.amblard_NOSPAM_wanadoo.fr (replace _NOSPAM_ by @)
//
//   This program is free software; you can redistribute it and/or modify 
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2 of the License, or
//   (at your option) any later version. 

/**
VOISINAGE. Cette classe représente un <i>nombre</i> appelé <i>limite</i> 
et une <i>position</i> par rapport à ce nombre.
La <i>position</i>  représente elle-même un ensemble de réels qui vérifient une relation avec la <i>limite</i>.<br>
<b>Opérateurs binaires</b> . Il est possible d'appliquer les opérateurs usuels aux instances de cette classe : le résultat 
est un nouveau voisinage. La limite du résultat est obtenue en appliquant l'opérateur aux limites des deux voisinages. La 
position du résultat est obtenue en appliquant l'opérateur aux ensembles décrits par les deux positions.<br>
<b>Opérateurs unaires et fonctions</b>. On procède comme pour les opérateurs binaires en calculant 
l'image de la limite et en appliquant l'opérateur ou la fonction à l'ensemble décrit par la position. 
Par convention, le résultat n'est pas défini lorsque cet ensemble 
n'est pas entièrement contenu dans le domaine de définition de la fonction.
@see posi
*/

public class vois extends calcul {

/*_______________________________________ VARIABLES ________________________________________*/

/**
Nombre représentant la limite.
*/
	nbre L ; 
	
/**
Position par rapport à la limite.
*/
	posi plim = new posi() ; 
	
/**
Chaîne contenant une information supplémentaire.
*/
	String info = "" ; 
	
/**
Indique si l'objet <i>box</i> renvoyé par la méthode <i>toBox()</i> de cette classe intègre l'information
concernant la position.
*/
	boolean showpos = false ;
	
/*______________________________________ CONSTRUCTION ______________________________________*/

/**
Construit un voisinage.
@param n nombre représentant la limite
@param p position par rapport à la limite
*/
//	public vois(nbre n , posi pl) { L = n ; plim = pl ; }
	
/**
Construit un voisinage.
@param f une formule représentant la limite
@param pl une position par rapport à la limite
*/
	public vois(form f , posi pl) { L = f.getNbre() ; plim = pl ; }
	
/**
Construit un voisinage à partir d'un voisinage existant.
@param k un voisinage
*/
	public vois(vois k) { L = new nbre(k.L) ; plim = new posi(k.plim) ; showpos = k.showpos ; }
	
/*________________________________________ MÉTHODES ________________________________________*/

// 1. TESTS _________________________________________________________________________________

/**
Indique si ce voisinage est défini.
*/
	boolean isDef() { return (L.estDef() && plim.isDef()) ; }

// 2. CHAÎNES & BOÎTES ______________________________________________________________________

/**
Renvoie une description de ce voisinage.
*/
	public String toString() { return "[" + L + " _ " + plim + "] " + info ; }
	
/**
Renvoie un objet <i>box</i> permettant de dessiner ce voisinage.
*/
	public box toBox() {
		box r = box.rightArrow(10f) ; r.centre("  ") ; r.append(L.toBox()) ;
		if (!showpos || plim.isAll()) return r ;
		String s = "" ; if (plim.right) s += "+" ;  if (plim.left) s += "-" ;
		box.redSize(.8f) ; box e = new box(s , false) ; box.prevSize() ;
		return box.boxPow(r , e) ;
	}

// 3. CALCULS _______________________________________________________________________________

/**
Somme.<br>
La limite du résultat est non définie en cas d'indétermination "∞ - ∞" .
@param a un voisinage
*/
	vois add(vois a) {
		vois r ;
		if (!isDef() || !a.isDef()) return vndef() ;
		if ((L.estEgal(pinf) && a.L.estEgal(minf)) || (L.estEgal(minf) && a.L.estEgal(pinf)))  r = vndef("∞ - ∞") ;
		else r =  new vois(L.Gadd(a.L) , plim.combine(a.plim)) ;
		return r ;
	}
	
/**
Différence.<br>
La limite du résultat est non définie en cas d'indétermination "∞ - ∞" .
@param a un voisinage
*/
	vois sub(vois a) {
		if (!isDef() || !a.isDef()) return vndef() ;
		return add(a.opp()) ;
	}
	
/**
Produit.<br>
La limite du résultat est non définie en cas d'indétermination "0 * ∞" .
@param a un voisinage
*/
	vois mult(vois a) {
		vois r ;
		if (!isDef() || !a.isDef()) return vndef() ;
		if ((a.L.estEgal(0) && L.estInfini()) || (L.estEgal(0) && a.L.estInfini())) return vndef("0 * ∞") ;
		form p = L.Gmult(a.L) ; posi pl = new posi() ;
		if (L.estStrictNeg()) {
			if (a.L.estStrictNeg()) pl = plim.opp().combine(a.plim.opp()) ;
			else if (a.L.estStrictPos()) pl = plim.opp().combine(a.plim).opp() ;
			else pl = a.plim.opp() ;
		}
		else if (L.estStrictPos()) {
			if (a.L.estStrictNeg()) pl = plim.combine(a.plim.opp().opp()) ;
			else if (a.L.estStrictPos()) pl = plim.combine(a.plim) ;
			else pl = a.plim ;
		}
		else {
			if (a.L.estStrictNeg()) pl = plim.opp() ;
			else if (a.L.estStrictPos()) pl = plim ;
			else pl = plim.multZZ(a.plim) ;
		}
		r = new vois(p , pl) ;
		return r ;
	}

/**
Quotient.<br>
La limite du résultat est non définie en cas d'indétermination "∞ / ∞" 
ou "0 / 0" ou " ... / 0" avec un signe inconnu au dénominateur.
@param a un voisinage
*/
	vois div(vois a) {
		vois r ;
		if (!isDef() || !a.isDef()) r = vndef() ;
		else if (L.estInfini() && a.L.estInfini()) r = vndef("∞ / ∞") ;
		else if (L.estEgal(0) && a.L.estEgal(0)) r = vndef("0 / 0") ;
		else if (a.L.estEgal(0) && (a.plim.exact || (a.plim.left && a.plim.right))) r = vndef("/0") ;
		else r = mult(a.inv()) ;
		return r ;
	}
	
	
/**
Opposé.
*/
	vois opp() { 
		if (!isDef()) return vndef() ;
		return new vois(L.Gopp() , plim.opp()) ; 
	}
	
/**
Inverse.<br>
La limite du résultat est non définie en cas d'indétermination " ... / 0" avec un signe inconnu au dénominateur.
*/
	vois inv() {
		vois r ;
		if (!isDef()) r = vndef() ;
		else if (L.estEgal(0)) {
			if (plim.exact || (plim.left && plim.right)) r = vndef("/0") ;
			else if (plim.left) r = vminf() ; 
			else r = vpinf() ;
		}
		else r = new vois(L.Ginv() , plim.opp()) ;
		return r ;
	}
	
/**
Puissance.
@param a un voisinage
*/
	vois pow(vois a) {
		if (!isDef() || !a.isDef()) return vndef() ;
		nbre n = a.L.getNbre() ; if (n.type == 2) return pow((int)n.val) ;
		vois r = a.mult(ln()).exp() ; 
		r.L = L.Gpow(a.L).getNbre() ; 
		return r ;
	}
	
/**
Puissance entière.
@param p un exposant entier
*/
	vois pow(int p) {
		if (p == 0) return new vois(new nbre(1) , posi.pEgal) ;
		if (p < 0) {
			if (L.estEgal(0)) return pow(-p).inv() ;
			else return inv().pow(-p) ;
		}
		vois r = new vois(L.Gpow(new nbre(p)) , new posi(plim)) ;
		if (p%2 == 0) {
			if (L.estEgal(0)) { if (plim.left) r.plim.right = true ; r.plim.left = false ; }
			else if (L.estStrictNeg()) r.plim = plim.opp() ;
		}
		return r ;
	}
		
/**
Indice.
@param a un voisinage
*/
	vois ind(vois a) { return new vois(L.Gind(a.L) , plim) ; }

/**
Image par la valeur absolue.
*/
	vois abs() {
		if (!isDef()) return vndef() ;
		vois r = new vois(L.Gftn("abs") , plim) ;
		if (L.estEgal(0)) r.plim.left = false ;
		else if (L.estStrictNeg()) r.plim = plim.opp() ;
		return r ;
	}
	
/**
Image par la racine carrée.
*/
	vois rac() {
		if (!isDef()) return vndef() ;
		if (L.estStrictNeg() || (L.estEgal(0) && plim.left)) return vndef("ndef") ;
		return new vois(L.Gftn("rac") , plim) ;
	}
	
/**
Image par le sinus.
*/
	vois sin() {
		if (!isDef()) return vndef() ; if (L.estInfini()) return vndef("nlim") ;
		return new vois(L.Gftn("sin") , posi.pAll) ;
	}
	
/**
Image par le cosinus.
*/
	vois cos() {
		if (!isDef()) return vndef() ; if (L.estInfini()) return vndef("nlim") ;
		return new vois(L.Gftn("cos") , posi.pAll) ;
	}
	
/**
Image par la tangente.
*/
	vois tan() {
		if (!isDef()) return vndef() ; if (L.estInfini()) return vndef("nlim") ;
		return  new vois(L.Gftn("tan") , plim) ;
	}
	
/**
Image par la fonction <i>arcsin</i>.
*/
	vois asin() {
		if (!isDef()) return vndef() ;
		vois r = new vois(L.Gftn("asin") , plim) ;
		return r ;
	}
	
/**
Image par la fonction <i>arcos</i>.
*/
	vois acos() {
		if (!isDef()) return vndef() ;
		vois r = new vois(L.Gftn("acos") , plim.opp()) ;
		return r ;
	}
	
/**
Image par la fonction <i>arctan</i>.
*/
	vois atan() {
		if (!isDef()) return vndef() ;
		vois r = new vois(L.Gftn("atan") , plim) ;
		return r ;
	}
	
/**
Image par le logarithme népérien.
*/
	vois ln() {
		if (!isDef()) return vndef() ;
		if (L.estStrictNeg()) return vndef("ndef") ;
		if (L.estEgal(0)) {
			if (!plim.left && !plim.exact) return vminf() ;
			else return vndef("ndef") ;
		}
		return new vois(L.Gftn("ln") , plim) ;
	}
	
/**
Image par le logarithme décimal.
*/
	vois log() {
		if (!isDef()) return vndef() ;
		if (L.estStrictNeg()) return vndef("ndef") ;
		if (L.estEgal(0)) {
			if (!plim.left && !plim.exact) return vminf() ;
			else return vndef("ndef") ;
		}
		return new vois(L.Gftn("log") , plim) ;
	}	
	
/**
Image par l'exponentielle.
*/
	vois exp() {
		if (!isDef()) return vndef() ;
		return  new vois(L.Gftn("exp") , plim) ;
	}
	
/**
Image par la partie entière.
*/
	vois ent() {
		if (!isDef()) return vndef() ;
		double d = Math.floor(L.val) ;
		if(d != L.val) return new vois(new nbre(d) , posi.pEgal) ;
		if (plim.left && !plim.exact && !plim.right) return new vois(new nbre(d - 1) , posi.pEgal) ;
		if (!plim.left) return new vois(new nbre(d) , posi.pEgal) ;
		return vndef() ;
	}
	
/**
Image par la dérivée de la valeur absolue.
*/
	vois dabs() {
		if (!isDef()) return vndef() ;
		double d  = (L.estStrictNeg()) ? -1 :
					(L.estStrictPos()) ?  1 :
					(plim.left && !plim.exact && !plim.right) ? -1 :
					(!plim.left && plim.exact && !plim.right) ?  0 :
					(!plim.left && !plim.exact && plim.right) ?  1 :
					Double.NaN ;
		return new vois(new nbre(d) , posi.pEgal) ;
	}
	
/**
Image par la dérivée de la partie entière.
*/
	vois dent() {
		if (!isDef()) return vndef() ;
		double d = Math.floor(L.val) ;
		if(d != L.val) return new vois(new nbre(0) , posi.pEgal) ;
		if (plim.left && !plim.exact && !plim.right) return new vois(new nbre(0) , posi.pEgal) ;
		if (!plim.left) return new vois(new nbre(0) , posi.pEgal) ;
		return vndef() ;
	}
	
/**
Image par une fonction.
@param s le nom de la fonction
*/
	vois ftn(String s) {
		if (!isDef()) return vndef() ;
		vois r = new vois(L.Gftn(s) , posi.pAll) ;
		return r ;
	}
	
/*___________________________________ MÉTHODES STATIQUES ___________________________________*/

/**
Renvoie un voisinage non défini.
*/
	public static vois vndef() { return new vois(new nbre(Double.NaN) , new posi()) ; }
	
/**
Renvoie un voisinage non défini avec une information sur la raison de l'indétermination.
@param s une chaîne de caractères décrivant l'indétermination
*/
	public static vois vndef(String s) {
		vois r = new vois(new nbre(Double.NaN) , new posi()) ;
		r.info = s ;
		return r ;
	}
	
/**
Renvoie un voisinage de +∞.
*/
	public static vois vpinf() {
		return new vois(new nbre(Double.POSITIVE_INFINITY) , new posi(true , false , false)) ;
	}
	
/**
Renvoie un voisinage de -∞.
*/
	public static vois vminf() {
		return new vois( new nbre(Double.NEGATIVE_INFINITY) , new posi(false , false , true) ) ;
	}	
	
}
