//
//  Copyright (c) 2005 Joël Amblard - joel.amblard_NOSPAM_wanadoo.fr (replace _NOSPAM_ by @)
//
//   This program is free software; you can redistribute it and/or modify 
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2 of the License, or
//   (at your option) any later version. 

import java.util.Vector ;
import java.util.EmptyStackException ;
import java.awt.geom.Point2D ;

/**
FORMULES. Cette classe est la racine des objets utilisés dans les calculs.<br>
Une formule a trois champs : un opérateur <i>op</i> , une liste <i>F</i> et une chaîne <i>str</i>.<br>
La liste <i>F</i> contient les fils de la formule. 
L'opérateur <i>op</i> décrit l'opération qui doit être effectuée sur la liste <i>F</i> des fils. 
La chaîne  <i>str</i> est une représentation de la formule lorsque <i>op</i> et <i>F</i> ne sont pas instanciés.
Les formules sont de deux types : 
<ul>
	<li> évaluées : <i>op</i>  et <i>F</i>  ont la valeur <i>null</i>
	<li> non évaluées : i>op</i>  et <i>F</i> sont instanciés
</ul>
*/

public class form extends calcul {

/*_______________________________________ VARIABLES ________________________________________*/

/**
Opérateur.
*/
	oper op ;
	
/**
Fils.
*/
	liste F  ;
	
/**
Représentation de la formule par une chaîne de caractères.
*/
	String str = "" ;
	
/*______________________________________ CONSTRUCTION ______________________________________*/

/**
Crée une formule évaluée représentée par un chaîne vide.
*/
    public form() { }

/**
Crée une formule à partir de la pile <i>Opn</i> de la classe <b>calcul</b>.
@param o l'opérateur de la formule créée
*/
    public form (oper o) throws EmptyStackException {
        op = o ; 
		F = new liste(o.poids) ; 
		F.top() ; while (F.hasPrev()) { F.prev() ; F.setD((form)Opn.pop()) ; }
    }

/**
Crée une formule évaluée à partir d'une chaîne.
@param s la chaîne décrivant la formule créée
*/
    public form (String s) { str = s ; }
	
/**
Crée une formule non évaluée à partir d'un opérateur et d'une liste.
@param o l'opérateur de la formule créée
@param L la liste des fils de la formule créée
*/
    public form (oper o , liste L) { op = o ; F = L ; }

/**
Crée une formule à partir d'une formule existante.
@param f la formule recopiée
*/
    public form(form f) {
		if (f.op != null) { op = f.op.copie() ; F = f.F.copie() ; }
		else str = new String(f.str) ;
    }

/*________________________________________ MÉTHODES ________________________________________*/

// 2. TESTS _________________________________________________________________________________

/**
Détermine si la valeur numérique de cette formule est égale à une valeur numérique donnée.
*/
	boolean estEgal(double d) { return (getDoubleValue() == d) ; }

/**
Détermine si la valeur numérique de cette formule est strictement négative.
*/
	boolean estStrictNeg() { return (getDoubleValue() < 0) ; }
	
/**
Détermine si la valeur numérique de cette formule est strictement positive.
*/
	boolean estStrictPos() { return (getDoubleValue() > 0) ; }
	
/**
Détermine si la valeur numérique de cette formule est infinie.
*/
	boolean estInfini() { return Double.isInfinite(getDoubleValue()) ; } 

/**
Détermine si la valeur numérique de cette formule est définie.
*/
	boolean estDef()  { return !Double.isNaN(getDoubleValue()) ; }
		
/**
Détermine si l'opérateur de cette formule est une somme.
*/
	boolean operIsSum() { return ((op != null) && (op instanceof op_somme) && (F.length() > 0)) ; }
	
/**
Détermine si l'opérateur de cette formule est un produit.
*/
	boolean operIsProd() { return ((op != null) && (op instanceof op_produit) && (F.length() > 0)) ; }
	
/**
Détermine si l'opérateur de cette formule est un oppose.
*/
	boolean operIsOpp() { return ((op != null) && (op instanceof op_oppose) && (F.length() == 1)) ; }
	
/**
Détermine si l'opérateur de cette formule est un inverse.
*/
	boolean operIsInv() { return ((op != null) && (op instanceof op_inverse) && (F.length() == 1)) ; }
	
/**
Détermine si l'opérateur de cette formule est un quotient.
*/
	boolean operIsQuot() { return ((op != null) && (op instanceof op_quotient) && (F.length() == 2)) ; }
	
/**
Détermine si l'opérateur de cette formule est une fonction connue.
*/
	boolean operIsFC() { 
		if ((op == null) || (F.length() != 1)) return false ; 
		return (indice(op.nom , FONCTIONS_CONNUES) >= 0) ; 
	}
	
/**
Détermine si une variable est utilisée dans cette formule.
@param s le nom de la variable
*/
	boolean hasVar(String s) {
		boolean b = false ; 
		if (op == null) return s.equals(toString()) ;
		else {
			F.bot() ; while(F.hasNext()) { b = F.next().hasVar(s) ; if (b) break ; }
		}
		return b ;
	}

/**
Détermine si la fonction <i>racine carrée</i> est utilisée dans cette formule.
*/
	form hasSqrt() {
		form a = null ; if (op == null) return a ;
		if (op.nom.equals("rac")) return this ;
		else {
			F.bot() ; while(F.hasNext()) { a = F.next().hasSqrt() ; if (a != null) break ; }
		}
		return a ;
	}

/**
Détermine si la fonction <i>partie entière</i> est utilisée dans cette formule.
*/
	boolean hasInt() {
		if (op == null) return false ;
		if (op.nom.equals("ent")) return true ;
		else {
			F.bot() ; while(F.hasNext()) { if(F.next().hasInt()) return true ; }
		}
		return false ;
	}

// 3. COPIE & INFO __________________________________________________________________________
	
/**
Renvoie une copie de cette formule.
*/
	form copie() { return new form(this) ; }
	
/**
Renvoie une formule équivalente à cette formule.
*/
	form getForm() { return copie() ; }
	
/**
Renvoie la priorité de cette formule.
*/
	int getPrior() { if (op == null) return pri + 1 ; else return op.prior ; }
	
/**
Renvoie le nom de l'opérateur de cette formule.
*/
	String getPere() { if (op == null) return "" ; else return op.nom ; }

/**
Renvoie un nombre pour cette formule.
*/
	nbre getNbre() { return null ; }

/**
Renvoie un monôme pour cette formule.
*/
	mono getMono() { return null ; }
	
/**
Renvoie un polynôme pour cette formule.
*/
	poly getPoly() { return null ; }
	
/**
Renvoie une valeur numérique pour cette formule.
*/
	double getDoubleValue() { return Double.NaN ; }
	
/**
Renvoie un signe pour cette formule.
*/
	int getSign() { return (operIsOpp()) ? -1 : 1 ; }

// 4. CALCULS _______________________________________________________________________________

/**
Opposé de cette formule.
*/
	form Gopp() {
		if ((op != null) && (op instanceof op_oppose)) return F.first().copie() ;
		else return nopp(this) ; 
	}
	
/**
Inverse de cette formule.
*/
	form Ginv() { 
		if ((op != null) && (op instanceof op_inverse)) return new form(F.first()) ;
		else return ninv(this) ; 
	}
	
/**
Somme de cette formule et d'une autre formule.
@param f la formule ajoutée
*/
	form Gadd(form f) {
		if (f instanceof nbre) return ((nbre)f).Gadd(this) ;
		if (f instanceof mono) return ((mono)f).Gadd(this) ;
		if (f instanceof poly) return ((poly)f).Gadd(this) ;
		return nsom(this , f) ; 
	}
	
/**
Différence de cette formule et d'une autre formule.
@param f la formule soustraite
*/
	form Gsub(form f) { return Gadd(f.Gopp()) ; }

/**
Produit de cette formule et d'une autre formule.
@param f la formule par laquelle on multiplie
*/
	form Gmult(form f) {
		if (estEgal(1)) return new form(f) ;
		if (estEgal(-1)) return f.Gopp() ;
		if (f instanceof nbre) return ((nbre)f).Gmult(this) ;
		if (f instanceof mono) return ((mono)f).Gmult(this) ;
		if (f instanceof poly) return ((poly)f).Gmult(this) ;
		return npro(this , f) ; 
	}
	
/**
Division de cette formule par une autre formule.
@param f la formule par laquelle on divise
*/
	form Gdiv(form f) { return Gmult(f.Ginv()) ; }

/**
Puissance de cette formule par une autre formule.
@param f la formule exposant
*/
	form Gpow(form f) { return npow(this , f) ; }

/**
Puissance entière de cette formule.
@param n l'exposant
*/
    form Gpow(int n) { 
		if (n == 0) return new nbre(1) ;
		int e = n ; if (n < 0) e *= -1 ; form r = new form(this) ;
		for(int i = 1 ; i < e ; i++) r = r.Gmult(this) ;
		if (n > 0) return r ; else return r.Ginv() ;
	}

/**
Indice de cette formule par une autre formule.
@param f la formule indice
*/
    form Gind(form f) { return nind(this , f) ; }

/**
Image de cette formule par une fonction.
@param s le nom de la fonction
*/
    form Gftn(String s) { return nftn(s , this) ; }

// 5. CHAÎNES _______________________________________________________________________________

/**
Renvoie une chaîne de caractères décrivant cette formule.
*/
	public String toString() {
		if (op == null) return str ;
		else return op.toOrdString(F) ;
	}	
	
/**
Renvoie une chaîne de caractères décrivant cette formule entourée de
parenthèses lorsque la priorité l'exige.
@param p le niveau de priorité
*/
	String toString(int p) {
		String r = toString() ; if (getPrior() < p) r = "\u0028" + r + "\u0029" ; return r ;
	}
	
// 6. BOÎTES ________________________________________________________________________________

/**
Renvoie un objet <i>box</i> permettant de dessiner cette formule.
*/
	box toBox() {
		if (op == null) return new box(toString()) ; 
		if (operIsOpp()) {
			form a = F.first() ;
			if (a.operIsQuot()) {
				return a.op.toBox(new liste(a.F.first().Gopp() , a.F.next())) ; 
			}
		}
		return op.toBox(F) ;
	}
	
/**
Renvoie un objet <i>box</i> permettant de dessiner cette formule 
entouré de parenthèses lorsque la priorité l'exige.
@param p le niveau de priorité
*/
	box toBox(int p) { 
		box r = toBox() ; 
		if (getPrior() < p) {
			String s = toString() ;
			if ((s.indexOf('(') >= 0) || (s.indexOf(')') >= 0)) r.parenth("[" , "]") ; else r.parenth() ;
		}
		return r ; 
	}
	

// 7. SUBSTITUTIONS _________________________________________________________________________

/**
Substitue une formule à une variable.
@param s la variable
@param t la formule substituée
*/
	form subst(String s , form t) {
		form res ;
		if (op == null) {
			if (s.equals(toString())) res = t.copie() ;
			else return res = copie() ;
		}
		else {
			liste A = new liste() ; form a ;
			F.bot() ; while(F.hasNext()) A.add(F.next().subst(s , t)) ;
			res = new form(op , A) ;
		}
		return res.eval() ;
	}
	
/**
Substitue une formule à une formule.
@param r la formule recherchée
@param f la formule substituée à la formule recherchée
*/
	form subst(form r , form f) { return subst(r.toString() , f) ; }
	
// 8. CALCULS NUMÉRIQUES ____________________________________________________________________

/**
Renvoie une valeur numérique lorsque une variable a une valeur donnée.
@param s la variable
@param v la valeur de la variable
*/
	double toDouble(String s , double v) {
		if (op == null) return (s.equals(str)) ? v : valD(str) ;
		else {
			double[] res = new double[F.length()] ;
			for(int i = 0 ; i < F.length() ; i++) res[i] = (F.get(i)).toDouble(s , v) ;
			return op.toDouble(res) ;
		}
	}
	
/**
Renvoie une valeur numérique lorsque deux variables ont deux valeurs données.
@param s la première variable
@param v la valeur de la première variable
@param t la seconde variable
@param w la valeur de la seconde variable
*/
	double toDouble(String s , double v , String t , double w) {
		if (op == null) return (s.equals(str)) ? v : (t.equals(str)) ? w : valD(str) ;
		else {
			double[] res = new double[F.length()] ;
			for(int i = 0 ; i < F.length() ; i++) res[i] = F.get(i).toDouble(s , v , t , w) ;
			return op.toDouble(res) ;
		}
	}
	
/**
Renvoie un point dont les coordonnées sont une abscisse donnée et
une ordonnée déterminée par une variable et cette abscisse.
@param s la variable
@param v l'abscisse du point
*/
	Point2D.Float toPoint(String s , double v) {
		return new Point2D.Float((float)v , (float)toDouble(s , v)) ; 
	}
		
// 9. CALCULS SYMBOLIQUES ___________________________________________________________________

/**
Évaluation à un niveau indiqué.<br>
Les niveaux d'évaluation sont les suivants :
<ul>
  <li> NO_EVAL : aucune évaluation
  <li> EVAL_SIGNS : simplification des signes
  <li> EVAL_NBRES : calculs avec les nombres
  <li> EVAL_MONOS : calculs avec les monômes
  <li> EVAL_POLYS : calculs avec les polynômes
</ul>
@param l le niveau d'évaluation choisi
*/
	form eval(int l) {
		if (op != null) {
			liste R = new liste() ;
			F.bot() ; while(F.hasNext()) R.add(F.next().eval(l)) ;
			op.level = l ; 
			return op.eval(R) ;
		}
		else return new mono(str) ;
	}
	
	form eval() { return eval(EVAL_POLYS) ; }
	
/**
Extrait les numérateurs et les dénominateurs d'un produit et renvoie leur quotient.
*/
	form toQuot() {
		form  r = this ;
		if (op != null) {
			liste R = new liste() ;
			while (F.hasNext()) R.add(F.next().toQuot()) ;
			if (operIsProd()) {
				form a ; liste D = new liste() ; R.bot() ; while(R.hasNext()) {
					a = R.next() ; if (a.operIsInv()) { D.add(a.F.first()) ; R.remA() ; }
				}
				r = resultForQuotient(false , R , D) ;
			}
			else r = new form(op , R) ; 
		}
		return r ;
	}
        
// 10. DÉRIVÉE ______________________________________________________________________________

/**
Calcul de la dérivée.
@param v la variable par rapport à laquelle on dérive
*/
	form toDer(String v) {
		form res ;
		if (op == null) res = (v.equals(str)) ? new nbre(1) : new nbre(0) ;
		else {
			liste R = new liste() ;
			F.bot() ; while (F.hasNext()) {
				R.add(F.next().toDer(v)) ;
				form d = R.last() ; 
			}
			res = op.toDer(F , v , R) ;
		}
		return res ;
	}
	

// 11. LIMITES ______________________________________________________________________________

/**
Renvoie, si possible, une échelle de comparaison pour cette formule.
*/
	equg toEquivG(String sx , vois k) {
		equg r = null ; 
		if (op != null) {
			if ((op instanceof op_indice) && sx.equals(toString())) return new equg(new nbre(1) , sx , k) ;
			Vector V = new Vector() ; equg a ;
			F.bot() ; while(F.hasNext()) {
				a  = F.next().toEquivG(sx , k) ; 
				if (a == null) { V.clear() ; break ; } else V.add(a) ;
			}
			if (V.size() > 0) r = op.toEquivG(V) ;
		}
		if ((r != null) && (r.vl.L.estDef())) return r ; 
		else return null ;
	}
	
	
/**
Calcul d'un développement de Taylor.
@param sv la variable par rapport à laquelle on dérive
@param l le nombre près duquel développe
@param o l'ordre du développement
*/
	form taylorForm (String sv , nbre l , int o) {
		poly s = new poly(sv) ; s.Padd(l) ;
		form e  = (l.val == 0) ? copie() : subst(sv , s) ;
		nbre z = new nbre(0) ;
		liste L = new liste() ; 
		L.add(e.subst(sv , z)) ;
		form t ;
		for(int i = 1 ; i < o + 1 ; i++) {
			e = npro(e.toDer(sv) , new nbre(1 , i)) ;
			t = e.subst(sv , z) ;
			if (!t.estEgal(0)) L.add(npro(t , new mono(sv , i))) ;
		}
		return nsom(L).eval() ;
	}
	
/**
Remplace les fontions par un développement de Taylor.
@param sv la variable par rapport à laquelle on dérive
@param l le nombre où l'on développe
@param o l'ordre du développement
*/
	form taylor (String sv , nbre l , int o) {
		if (op == null) return copie().eval() ;
		else {
			if (op instanceof op_fonction) return taylorForm(sv , l , o) ;
			else {
				liste A = new liste() ;
				F.bot() ; while(F.hasNext()) A.add(F.next().taylor(sv , l , o)) ;
				return (new form(op , A)).eval() ;
			}			
		}
	}

/**
Calcul d'une limite en appliquant les règles sur les opérations.
@param sx la variable
@param k le mode de convergence de la variable
*/
	vois operLimit(String sx , vois k) {
		vois r = vois.vndef() ; if (Double.isNaN(k.L.val)) return r ;
		if (op == null) {
			if (sx.equals(toString())) return k ; else return vois.vndef() ;
		}
		else {
			Vector R = new Vector() ; 
			F.bot() ; while (F.hasNext()) R.add(F.next().operLimit(sx , k)) ;
			r = op.operLimit(R) ; 
			if (r == null) return vois.vndef() ; else return r ;
		}
	}
	
/**
Calcul d'une limite.<br>
On calule la limite  en appliquant d'abord les règles sur les opérations : si le résultat est défini,
il est renvoyé. Sinon , on calcule la limite en construisant un échelle de comparaison.
Dans tous les cas, la variable est supposée tendre vers 0 ou +∞ ou -∞.
@param sx la variable
@param k le mode de convergence de la variable
*/
	vois getLimit(String sx , vois k) {
		vois r = operLimit(sx , k) ; 
		if (!r.L.estDef() && (op != null)) {
			equg eq = toEquivG(sx , k) ;
			if (eq != null) r = eq.vl ; 
		}
		return r ;
	}
	
/**
Calcul d'une limite.<br>
On calule la limite en testant plusieurs modes de calcul. 
Si un mode de calcul aboutit, le résultat est renvoyé. Sinon on passe au suivant.
<ul>
  <li> calcul avec les règles sur les opérations
  <li> calcul avec les échelles de comparaison
  <li> calcul d'un développement de Taylor et application des règles sur les opérations
  <li> application des échelles de comparaison au développement de Taylor précédemment calculé
</ul>
@param sx la variable
@param k le mode de convergence de la variable
*/
	vois computeLimit(String sx , vois k) {
		vois r = vois.vndef() ; if (Double.isNaN(k.L.val)) return r ;
		form e ; vois l ;
		if (Double.isInfinite(k.L.val) || (k.L.val == 0)) {
			e = copie().eval(EVAL_MONOS) ; 
			l = k ;
		}
		else {
			poly s = new poly(sx) ; s.Padd(k.L) ;
			e = subst(sx , s).eval() ;
			l = new vois(new nbre(0) , k.plim) ;
		}
		r = e.getLimit(sx , l) ;
		if (!r.L.estDef() && (l.L.val == 0)) {
			e = e.taylor(sx , l.L , 3).eval() ;
			r = e.getLimit(sx , l) ;
		}
		return r ;
	}
		
}