//
//  Copyright (c) 2005 Joël Amblard - joel.amblard_NOSPAM_wanadoo.fr (replace _NOSPAM_ by @)
//
//   This program is free software; you can redistribute it and/or modify 
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2 of the License, or
//   (at your option) any later version. 

/** ÉCHELLES DE COMPARAISON. 
Cette classe fournit le moyen de lever (dans les cas simples) des indéterminations provenant 
du calcul des limites. <br>
On considère une fonction f(<i>x</i>) définie sur un voisinage V de 0 ou de +∞ ou de -∞ .
Une échelle est construite lorsqu'il est possible de définir une fonction : <br>
<center> g(<i>x</i>) = (ln <i>x</i>)<sup>a</sup> <i>x</i><sup>b</sup> e<sup>c<i>x</i></sup> h(x)
</center><br>
qui coïncide avec f(<i>x</i>) sur le voisinage V et pour laquelle la limite de h(<i>x</i>) 
 est un réel fini et non nul. La fonction : <br>
<center> m(<i>x</i>) = (ln <i>x</i>)<sup>a</sup> <i>x</i><sup>b</sup> e<sup>c<i>x</i></sup>
</center><br>
est appelée "partie principale" et la fonction h est appelée "facteur".
*/

public class equg extends calcul {

/*_______________________________________ VARIABLES ________________________________________*/

/**
Nom de la variable.
*/
	String sx ;
	
/**
Voisinage de la variable.
*/
	vois vx ;
	
/**
Exposant du logarithme.
*/
	nbre lnpart = new nbre(0) ;
	
/**
Exposant du monôme.
*/
	nbre monopart = new nbre(0) ;
	
/**
Exposant de l'exponentielle.
*/
	nbre exppart = new nbre(0) ;
	
/**
Formule représentant la partie principale.
*/
	form main ;
	
/**
Formule représentant le facteur.
*/
	form factor ;
	
/**
Formule représentant le produit de la partie principale et du facteur.
*/
	form replace ;
	
/**
Voisinage décrivant la limite de la partie principale.
*/
	vois vm ;
	
/**
Voisinage décrivant la limite du facteur.
*/
	vois vf ;
	
/**
Voisinage décrivant la limite de cette échelle.
*/
	vois vl ;
	
/*______________________________________ CONSTRUCTION ______________________________________*/

/**
Construit une échelle à partir d'une échelle existante.
*/
	public equg(equg E) {
		sx = new String(E.sx) ; 
		vx = new vois(E.vx) ;
		factor = E.factor.copie() ;
		lnpart = new nbre(E.lnpart) ; 
		monopart = new nbre(E.monopart) ; 
		exppart = new nbre(E.exppart) ;
		vm = new vois(E.vm) ; 
		vf = new vois(E.vf) ; 
		vl = new vois(E.vl) ; 
	}

/**
Construit une échelle à partir d'un nombre.
@param n un nombre
@param s une chaîne représentant la variable
@param k un voisinage
*/
	public equg(nbre n , String s , vois k) { 
		sx = new String(s) ; 
		vx = new vois(k) ; 
		factor = new nbre(n) ; 
		computeLimit() ;
	}
	
/**
Construit une échelle à partir d'un monôme et d'un voisinage.
@param m un monôme
@param k un voisinage
*/
	public equg(mono m , vois k) {
		sx = new String(m.base) ; 
		vx = new vois(k) ; 
		poly pr ;
		monopart = new nbre(m.deg) ; 
		factor = new nbre(m.coeff) ;
		computeLimit() ;
	}

/**
Construit une échelle à partir d'un polynôme et d'un voisinage.
@param p un polynôme
@param k un voisinage
*/
	public equg(poly p , vois k) {
		sx = new String(p.base) ; 
		vx = new vois(k) ;
		int deg = (vx.L.val == 0) ? p.deg - p.C.deg() : p.deg ;
		monopart = new nbre(deg) ;
		poly pr = new poly(p) ; 
		pr.maj(pr.C , p.deg - deg) ;
		factor = pr ;
		computeLimit() ;
	}
	
/*________________________________________ MÉTHODES ________________________________________*/

// 1. MISE À JOUR __________________________________________________________________________

/**
Calcule les limites.
*/
	void computeLimit() {
		form fm =  ((exppart.val != 0) && vx.L.estInfini()) ? nftn("exp" , new mono(exppart , sx)) :
					(monopart.val != 0) ? npow(new mono(sx) , monopart) :
					(lnpart.val != 0) ? npow(nftn("ln" , new mono(sx)) , lnpart) :
					new nbre(1) ;
		vm = fm.operLimit(sx , vx) ;
		vf = factor.operLimit(sx , vx) ;
		vl = vm.mult(vf) ;
		//lim = vl.L ;
	}
	
/**
Calcule la formule représentant cette échelle.
*/
	void setForm() {
		liste R = new liste() ;
		if (exppart.val != 0) R.add(nftn("exp" , new mono(exppart , sx))) ;
		if (monopart.val != 0) {
			double d = Math.abs(monopart.val) , f = d - Math.floor(d) ; form fm ;
			if ((f == .5) || (f == 0)) {
				d -= f ;
				fm = (d > 0) ? new mono(sx , (int)d) : null ;
				if (f == .5) fm = (fm != null) ? npro(fm , nftn("rac" , new mono(sx))) : nftn("rac" , new mono(sx)) ;
			}
			else fm = npow(new mono(sx) , monopart) ;
			if (monopart.val < 0) fm = fm.Ginv() ;
			R.add(fm) ;
		}
		if (lnpart.val != 0) R.add(nftn("ln" , new mono(lnpart , sx))) ;
		if (R.length() == 0) main = new nbre(1) ;
		else if (R.length() == 1) main = R.first() ;
		else main = npro(R) ;
		form r ;
		if (main.estEgal(1)) replace = factor.copie() ; 
		else {
			replace = npro(main , factor) ; ((op_produit) replace.op).boxquot = false ;
		}
	}

// 2. TESTS _________________________________________________________________________________

/**
Compare cette échelle à une autre échelle.
@param a une échelle de comparaison
*/
	int compareTo(equg a) {
		double d = Double.NaN ;
		if (vx.L.estEgal(0)) {
			if ((monopart.val != 0) || (a.monopart.val != 0)) d = a.monopart.val - monopart.val ;
			else if ((lnpart.val != 0) || (a.lnpart.val != 0)) d = lnpart.val - a.lnpart.val ;
		}
		else {
			if ((exppart.val != 0) || (a.exppart.val != 0)) d = exppart.val - a.exppart.val ;
			else if ((monopart.val != 0) || (a.monopart.val != 0)) d = monopart.val - a.monopart.val ;
			else if ((lnpart.val != 0) || (a.lnpart.val != 0)) d = lnpart.val - a.lnpart.val ;
		}
		if (d > 0) return 1 ; else if (d < 0) return -1 ; else return 0 ;
	}
	
// 3. COPIE & INFO __________________________________________________________________________
	
/**
Calcule la partie principale et renvoie une formule représentant cette échelle.
*/
	form getForm() {
		liste R = new liste() ;
		if (exppart.val != 0) R.add(nftn("exp" , new mono(exppart , sx))) ;
		if (monopart.val != 0) {
			double d = Math.abs(monopart.val) , f = d - Math.floor(d) ; form fm ;
			if ((f == .5) || (f == 0)) {
				d -= f ;
				fm = (d > 0) ? new mono(sx , (int)d) : null ;
				if (f == .5) fm = (fm != null) ? npro(fm , nftn("rac" , new mono(sx))) : nftn("rac" , new mono(sx)) ;
			}
			else fm = npow(new mono(sx) , monopart) ;
			if (monopart.val < 0) fm = fm.Ginv() ;
			R.add(fm) ;
		}
		if (lnpart.val != 0) R.add(nftn("ln" , new mono(lnpart , sx))) ;
		if (R.length() == 0) main = new nbre(1) ;
		else if (R.length() == 1) main = R.first() ;
		else main = npro(R) ;
		form r ;
		if (main.estEgal(1)) r = factor.copie() ; 
		else {
			r = npro(main , factor) ; ((op_produit) r.op).boxquot = false ;
		}
		return r ;
	}

/**
Renvoie une chaîne de caractères décrivant cette échelle.
*/
	public String toString() { setForm() ; return "[" + main + "] "  + " [" + factor + "] " ; }

// 4. CALCULS _______________________________________________________________________________

/**
Opposé de cette échelle.
*/
	equg oppg() {
		equg r = new equg(this) ;
		r.factor = r.factor.Gopp() ;
		r.computeLimit() ; 
		return r ;
	}

/**
Inverse de cette échelle.
*/
	equg invg() {
		equg r = new equg(this) ;
		r.lnpart.Nopp() ; 
		r.monopart.Nopp() ; 
		r.exppart.Nopp() ; 
		r.factor = r.factor.Ginv() ;
		r.computeLimit() ; 
		return r ;
	}

/**
Produit de cette échelle par une autre échelle.
@param a une échelle de comparaison
*/
	equg multg(equg a) {
		equg r = new equg(this) ;
		r.lnpart.Nadd(a.lnpart) ; 
		r.monopart.Nadd(a.monopart) ; 
		r.exppart.Nadd(a.exppart) ;
		liste R = new liste() ;
		if (factor.operIsProd()) R.add(factor.F) ; else R.add(factor) ;
		if (a.factor.operIsProd()) R.add(a.factor.F) ; else R.add(a.factor) ;
		r.factor = npro(R).eval() ;
		r.computeLimit() ; 
		return r ;
	}

/**
Modifie la partie principale de cette échelle en ajoutant trois nombres.
@param nln le nombre ajouté à l'exposant du logarithme
@param nmono le nombre ajouté à l'exposant du monome
@param nexp le nombre ajouté à l'exposant de l'exponentielle
*/
	void multn(nbre nln , nbre nmono , nbre nexp) {
		lnpart.Nadd(nln) ; 
		monopart.Nadd(nmono) ; 
		exppart.Nadd(nexp) ;
		computeLimit() ; 
	}

/**
Quotient de cette échelle par une autre échelle.
@param a une échelle de comparaison
*/
	equg quog(equg a) {
		equg r = new equg(this) ;
		r.lnpart.Nsub(a.lnpart) ; 
		r.monopart.Nsub(a.monopart) ; 
		r.exppart.Nsub(a.exppart) ;
		r.factor = (a.factor.estEgal(1)) ? factor :  factor.Gdiv(a.factor) ;
		r.computeLimit() ; 
		return r ;
	}

/**
Modifie la partie principale de cette échelle en soustrayant trois nombres.
@param nln le nombre soustrait à l'exposant du logarithme
@param nmono le nombre soustrait à l'exposant du monome
@param nexp le nombre soustrait à l'exposant de l'exponentielle
*/
	void quon(nbre nln , nbre nmono , nbre nexp) {
		lnpart.Nsub(nln) ; 
		monopart.Nsub(nmono) ; 
		exppart.Nsub(nexp) ;
		computeLimit() ; 
	}

/**
Puissance entière de cette échelle.
@param n un entier représentant l'exposant
*/
	equg powg(nbre n) {
		equg r = new equg(this) ; if (n.val == 1) return r ;
		r.lnpart.Nmult(n) ; 
		r.monopart.Nmult(n) ; 
		r.exppart.Nmult(n) ;
		nbre e = new nbre(n) ; if (n.val < 0) e.Nopp() ;
		r.factor = ((e.val != 1) ? npow(factor , e) : factor).eval() ; 
		if (n.val < 0) r.factor = r.factor.Ginv() ;
		r.computeLimit() ; 
		return r ;
	}

/**
Puissance de cette échelle par une autre échelle.
@param a une échelle de comparaison
*/
	equg powg(equg a) {
		form f = a.getForm().eval() ;
		if (! (f instanceof nbre)) return null ;
		else return powg((nbre)f) ;
	}

/**
Image de cette échelle par le logarithme népérien.
*/
	equg ln() {
		form f = getForm().eval() ;
		if (f instanceof nbre) return new equg((nbre)f.Gftn("ln") , sx , vx) ;
		if (!lnpart.estEgal(0)) return null ;
		equg r = new equg(this) ; 
		nbre nln = new nbre(monopart) , nmono = new nbre(exppart) , nexp = new nbre(0) ; 
		liste R = new liste(new nbre(1)) ;
		if (!nmono.estEgal(0)) {
			r.main = npow(new mono(sx) , nmono) ; 
			r.monopart = nmono ; 
			r.lnpart = r.exppart = new nbre(0) ;
			if (!nln.estEgal(0)) {
				R.add(npro(nln , nftn("ln" , new mono(sx))).Gdiv(r.main)) ; 
				nln = new nbre(0) ; 
			}
		}
		else if (!nln.estEgal(0)) {
			r.main = nftn("ln" , new mono(nln , sx)) ; 
			r.monopart = r.exppart = new nbre(0) ;
		}
		else return null ;
		if (!factor.estEgal(1)) R.add(nftn("ln" , factor).Gdiv(r.main)) ;
		r.factor = ((R.length() == 1) ? R.first() : nsom(R)).eval() ;
		return r ;
	}

/**
Image de cette échelle par le logarithme décimal.
*/
	equg log() {
		equg r = ln() ; if (r == null) return null ;
		r.factor.Gdiv(nftn("ln" , new nbre(10))) ;
		return r ;
	}
	
/**
Image de cette échelle par l'exponentielle.
*/
	equg exp() {
		form f = getForm() ;
		if (f instanceof nbre) return new equg((nbre)f.Gftn("exp") , sx , vx) ;
		if (!exppart.estEgal(0) || !factor.estEgal(1)) return null ;
		equg r = new equg(this) ; r.factor = new nbre(1) ;
		r.monopart = new nbre(lnpart) ; r.exppart = new nbre(monopart) ; r.lnpart = new nbre(0) ;
		return r ;
	}

}
