//
//  Copyright (c) 2005 Joël Amblard - joel.amblard_NOSPAM_wanadoo.fr (replace _NOSPAM_ by @)
//
//   This program is free software; you can redistribute it and/or modify 
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2 of the License, or
//   (at your option) any later version. 

import java.awt.* ;
import java.awt.font.FontRenderContext ;
import java.awt.geom.AffineTransform ;
import java.awt.geom.GeneralPath ;
import java.awt.geom.PathIterator ;
import java.awt.geom.Point2D ;
import java.text.DecimalFormat ;
import java.text.NumberFormat ;
import java.text.ParseException ;
import java.util.Locale ;
import java.util.ResourceBundle ;
import java.util.StringTokenizer ;
import java.util.Vector ;


public class dessin {

/*___________________________________________________________________________________________________________/
/   1. FENÊTRE ET REPÈRES ASSOCIÉS.																			 /
/___________________________________________________________________________________________________________*/

// 1.1 Fenêtre
/** 
* Largeur de la fenêtre dans laquelle on dessine.
*/
	static int W = 600 ;
	
/** 
* Hauteur de la fenêtre dans laquelle on dessine.
*/
	static int H = 400 ;

// 1.2 Repère-utilisateur
/** 
* Point-écran représentant l'origine du repère-utilisateur (le centre de la fenêtre par défaut) .
*/	
	static Point2D.Float ORIGIN = new Point2D.Float(W/2 , H/2) ;
	
/** 
* Dimension d'un pixel : 1/UNIT est le nombre de pixels par cm. 
* Par défaut, l'unité sur chaque axe du repère-utilisateur est le centimètre.
*/
	public final static double UNIT = .02845274 ;

/** 
* Unité de l'axe des abscisses : facteur par lequel UNIT est divisé. 
*/
	static double UX = 1 ;

/** 
* Unité de l'axe des ordonnées : facteur par lequel UNIT est divisé. 
*/
	static double UY = 1 ;

/** 
* Abscisse gauche du repère-utilisateur . 
*/
	static double XG = - ORIGIN.x * UNIT / UX ;

/** 
* Abscisse droite du repère-utilisateur . 
*/
	static double XD = (W - ORIGIN.x) * UNIT / UX ;

/** 
* Ordonnée basse du repère-utilisateur . 
*/
	static double YB = (ORIGIN.y - H) * UNIT / UY ;

/** 
* Ordonnée haute du repère-utilisateur . 
*/
	static double YH = ORIGIN.y * UNIT / UY ;

/** 
* Transformation affine qui permet de passer des coordonnées-utilisateur aux coordonnées-écran. <br>
* Sachant  :
* <ul> <li>O(X<sub>O</sub>,Y<sub>O</sub>) et M(X,Y) dans le repère-écran </li>
*		<li> M(x,y) dans le repère-utilisateur </li>
* </ul>
* on a les relations suivantes : <br>
* <li> X = x * UX / UNIT				+   X<sub>O</sub> </li>
* <li> Y =			-   y * UY / UNIT	+   Y<sub>O</sub> </li>
* </ul>
*/
	static AffineTransform 
		TO_SCREEN = new AffineTransform(UX / UNIT , 0 , 0 , -UY / UNIT , ORIGIN.x , ORIGIN.y) ;

// 1.3 Construction de points
/** 
* Point-écran représentant la souris . 
*/
	static Point2D.Float MOUSE = new Point2D.Float() ;

/** 
* Point-écran associé à un couple de coordonnées-utilisateur.
* @param x abscisse du point dans le repère-utilisateur
* @param y ordonnée du point dans le repère-utilisateur
*/
	static Point2D.Float screenPoint(double x , double y) { 
		return new Point2D.Float((float)(ORIGIN.x + x * UX / UNIT) , (float)(ORIGIN.y - y * UY / UNIT)) ; 
	}
	

/** 
* Pas de la grille sur laquelle sont placés les points-utilisateurs. <br>
*/
	static double USER_GRID = .1 ;

/** 
* Arrondi d'un résultat numérique. <br>
* Le résultat est arrondi de façon à être un multiple entier de <i> d </i> .
* @param x valeur numérique
* @param d le nombre dont le résultat doit être un multiple
*/
	static double round(double x , double d) {
		if (d <= 0) return x ;
		else return d * Math.rint(x / d) ; 
	}

/** 
* Transforme un couple de coordonnées-écran en un point-utilisateur qui est placé sur la grille.
* @param X une abscisse-écran
* @param Y une ordonnée-écran
*/
	static Point2D.Float gridPoint(double X , double Y) { 
		return new Point2D.Float((float) absG(X) , (float) ordG(Y)) ; 
	}
	
/** 
* Renvoie le point-utilisateur qui est placé sur la grille et qui est associé aux coordonnées de la souris.
*/
	static Point2D.Float gridPoint() { return gridPoint(MOUSE.x , MOUSE.y) ; }
	
/** 
* Renvoie le point-utilisateur associé aux coordonnées de la souris.
*/
	static Point2D.Float mousePoint() { 
		return new Point2D.Float((float) abs(MOUSE.x) , (float) ord(MOUSE.y)) ; 
	}
	
/** 
* Renvoie le point-utilisateur associé aux coordonnées de la souris qui ont été enregistrées 
* quand l'utilisateur a appuyé sur celle-ci.
*/
	static Point2D.Float startPoint() { 
		return new Point2D.Float((float) abs(MOUSE_START.x) , (float) ord(MOUSE_START.y)) ; 
	}
	
// 1.4 calcul de coordonnées

/** 
* Abscisse-écran correspondant à une abscisse-utilisateur.
* @param x l'abscisse dans le repère-utilisateur
*/	
	static int absE (double x) {return (int)Math.rint(W * (x - XG) / (XD - XG)) ; }

/** 
* Ordonnée-écran correspondant à une ordonnée-utilisateur.
* @param y l'ordonnée dans le repère-utilisateur
*/	
	static int ordE (double y) {return (int)Math.rint(H * (y - YH) / (YB - YH)) ; }

/** 
* Abscisse-utilisateur correspondant à une abscisse-écran.
* @param X l'abscisse dans le repère-utilisateur
*/	
	static double abs (double X) {return (XG + X * (XD - XG) / W) ; }

/** 
* Ordonnée-utilisateur correspondant à une ordonnée-écran.
* @param Y l'ordonnée dans le repère-utilisateur
*/	
	static double ord (double Y) {return (YH + Y * (YB - YH) / H) ; }
  
/** 
* Abscisse-utilisateur correspondant à une abscisse-écran. <br>
* La valeur est ajustée pour que le point soit sur la grille.
* @param X l'abscisse dans le repère-utilisateur
*/	
	static double absG (double X) { return round(abs(X) , USER_GRID) ; }

/** 
* Ordonnée-utilisateur correspondant à une ordonnée-écran. <br>
* La valeur est ajustée pour que le point soit sur la grille.
* @param Y l'ordonnée dans le repère-utilisateur
*/	
	static double ordG (double Y) { return round(ord(Y) , USER_GRID) ; }
  
	
/*___________________________________________________________________________________________________________/
/   2. TRANSLATION DU REPÈRE-UTILISATEUR.																			 /
/___________________________________________________________________________________________________________*/

/** 
* Point-écran mémorisant l'origine du repère-utilisateur au début d'une translation.
*/	
	static Point2D.Float ORIGIN_START = new Point2D.Float(W/2 , H/2) ;
	
/** 
* Point-écran mémorisant la position de la souris au début d'une translation. 
*/
	static Point2D.Float MOUSE_START = new Point2D.Float() ;
	
/** 
* Enregistre la position de l'origine et celle de la souris. <br>
* Tous les points libres sont invités à enregistrer leur position du fait de la possibilité d'un déplacement. 
*/
	static void mouseStart(int X , int Y) {
		ORIGIN_START.setLocation(ORIGIN) ; MOUSE_START.setLocation(X , Y) ;
		for(int i = 0 ; i < Z.GR.size() ; i++) ((objetG)Z.GR.elementAt(i)).savePoint() ;
	}
	
/** 
* Cette méthode est appelée lorsque la souris a été pressée sur un axe du repère utilisateur puis glissée.
* La différence entre la position actuelle de la souris et celle qu'elle occupait lorsqu'elle a été pressée 
* permet de translater le du repère-utilisateur.
*/
	static void moveOrigin() {
		ORIGIN.setLocation(ORIGIN_START.x + MOUSE.x - MOUSE_START.x , ORIGIN_START.y + MOUSE.y - MOUSE_START.y) ; 
		TO_SCREEN.setTransform(UX / UNIT , 0 , 0 , -UY / UNIT , ORIGIN.x , ORIGIN.y) ;
		pathWindow() ;
		if (Z.quadrillageVisible) pathGrid() ;
		if (Z.repereVisible) pathAxis() ;
		P.refresh() ;
		D.refresh() ;
		//Z.repaint() ;
	}
	
/** 
* Déplace le point <i> p </i> d'une distance <i> d </i> dans la direction de l'angle <i> a </i> .  <br>
* Le plan est orienté dans le sens trigonométrique direct et l'angle mesuré en degrés.
*/	
	static void movePoint(Point2D p , double d , double a) {
		double e = Math.PI * a / 180d ;
		p.setLocation(
			p.getX() + d*Math.cos(e) ,
			p.getY() + d*Math.sin(e)
		) ;
	}

/** 
* Renvoie le point situé à la distance <i> d </i> de <i> p </i> dans la direction de l'angle <i> a </i> .  <br>
* Le plan est orienté dans le sens trigonométrique direct et l'angle mesuré en degrés.
*/	
	static Point2D.Float pointMove(Point2D p , double d , double a) {
		double e = Math.PI * a / 180d ;
		return new Point2D.Float(
			(float)(p.getX() + d*Math.cos(e)) ,
			(float)(p.getY() + d*Math.sin(e))
		) ;
	}
	
/** 
* Renvoie la direction du vecteur d'origine <i> a </i> et d'extrêmité <i> b </i> . <br>
* Le plan est orienté dans le sens trigonométrique direct et l'angle mesuré en degrés.
*/	
	static double getDegrees(Point2D a , Point2D b) {
		double d = a.distance(b) ; if (d == 0) return Double.NaN ;
		double r = Math.acos((b.getX() - a.getX()) / d) ; if (b.getY() < a.getY()) r *= -1 ;
		return  r * 180d / Math.PI ;
	}
		
/** 
* Renvoie la direction du vecteur d'origine <i> a </i> et d'extrêmité <i> b </i> . <br>
* Le plan est orienté dans le sens trigonométrique direct et l'angle mesuré en radians.
*/	
	static double getRadians(Point2D a , Point2D b) {
		double d = a.distance(b) ; if (d == 0) return Double.NaN ;
		double r = Math.acos((b.getX() - a.getX()) / d) ; if (b.getY() < a.getY()) r *= -1 ;
		return  r ;
	}
		
/*___________________________________________________________________________________________________________/
/   3. MODIFICATION DES COORDONNÉES.																			 /
/___________________________________________________________________________________________________________*/

/** 
* Analyse un sytème de nouvelles coordonnées.
*/
	public static boolean parseCoords(String[] sf) {
		double[] nval = new double[] {XG , XD , YB , YH} ; boolean change = false ; 
		double x ; 
		for(int i = 0 ; i < 4 ; i++) {
			x = dblv(sf[i]) ; if (def(x) && (x != nval[i])) { nval[i] = x ; change = true ; }
		}
		if (!change || (nval[0] >= nval[1]) || (nval[2] >= nval[3])) return false ;
		XG = nval[0] ; XD = nval[1] ; YB = nval[2] ; YH = nval[3] ;
		UX = W * UNIT / (XD - XG) ;
		UY = H * UNIT / (YH - YB) ;
		if (UX * QX / UNIT < 10) QX = 10 * UNIT / UX ;
		if (UY * QY / UNIT < 10) QY = 10 * UNIT / UY ;
		return true ;		
	}
	
/** 
* Analyse un sytème de nouvelles unités.
*/
	public static boolean parseUnits(String[] sf) {
		double[] nval = new double[] {UX , UY} ; boolean change = false ; 
		double x ;
		for(int i = 0 ; i < 2 ; i++) {
			x = dblv(sf[i]) ; if (def(x) && (x != nval[i])) { nval[i] = x ; change = true ; }
		}
		if (!change || (nval[0] <= 0) || (nval[1] <= 0)) return false ;
		UX = nval[0] ; UY = nval[1] ;
		XG = -ORIGIN.x * UNIT / UX ;
		XD = (W - ORIGIN.x) * UNIT / UX ;
		YB = (ORIGIN.y - H) * UNIT / UY ;
		YH = ORIGIN.y * UNIT / UY ;
		if (UX * QX / UNIT < 10) QX = 10 * UNIT / UX ;
		if (UY * QY / UNIT < 10) QY = 10 * UNIT / UY ;
		return true ;		
	}
	
/** 
* Analyse un sytème de nouvelles unités du quadrillage.
*/
	public static boolean parseGrid(String[] sf) {
		double[] nval = new double[] {QX , QY} ; boolean change = false ; 
		double x ;
		for(int i = 0 ; i < 2 ; i++) {
			x = dblv(sf[i]) ; if (def(x) && (x != nval[i])) { nval[i] = x ; change = true ; }
		}
		if (!change || (nval[0] <= 0) || (nval[1] <= 0)) return false ;
		QX = nval[0] ; QY = nval[1] ;
		if (UX * QX / UNIT < 10) QX = 10 * UNIT / UX ;
		if (UY * QY / UNIT < 10) QY = 10 * UNIT / UY ;
		return true ;
	}
	
/** 
* Analyse un sytème de nouveaux arrondis.
*/
	public static boolean parseRound(String[] sf) {
		boolean change = false ; 
		double x ;
		x = dblv(sf[0]) ; if (def(x) && (x > 0) && (x != USER_GRID)) { USER_GRID = x ; change = true ; }
		x = dblv(sf[1]) ; if (def(x) && (x > 0) && (x != ROUND)) {
			ROUND = x ; change = true ;
			DIGITS = (x < 1) ? -(int)Math.rint(Math.log(x) / Math.log(10)) : (int) x ;
			NF.setMaximumFractionDigits(DIGITS) ;
		}
		return change ;
	}
	
/** 
* Enregistre un nouveau système de coordonnées.<br>
* Les unités ne sont pas recalculées.
* L'origine est mise à jour. 
* @param nval tableau contenant les nouvelles valeurs de XG , XD , YB , YH.
*/
	static void coords(double[] nval) {
		XG = nval[0] ; XD = nval[1] ; 
		YB = nval[2] ; YH = nval[3] ;
		if (UX * QX / UNIT < 10) QX = 10 * UNIT / UX ;
		if (UY * QY / UNIT < 10) QY = 10 * UNIT / UY ; 
		setOrigin() ;
	}
	
/** 
* Enregistre les nouvelles unités et les nouvelles coordonnées.<br>
* @param nval : tableau contenant les nouvelles valeurs de XG , XD , YB , YH.
* @param b les unités sont recalculées lorsque ce paramètre vaut <i>true</i>.
*/
	static void coords(double[] nval , boolean b) {
		if (b) {
			UX = W * UNIT / (nval[1] - nval[0]) ;
			UY = H * UNIT / (nval[3] - nval[2]) ;
		}
		coords(nval) ;
	}
	
/** 
* Zoom avant. Les unités sont multipliées par 3/4.
*/
	static void zoomPlus() {
		coords(
			new double[] {
				(7*XG + XD) / 8 , (7*XD + XG) / 8 , (7*YB + YH) / 8 , (7*YH + YB) / 8
			} ,
			true
		) ;
	}
	
/** 
* Zoom arrière. Les unités sont multipliées par 4/3.
*/
	static void zoomMoins() {
		coords( 
			new double[] {
				(7*XG - XD) / 6 , (7*XD - XG) / 6 , (7*YB - YH) / 6 , (7*YH - YB) / 6
			} ,
			true
		) ;
	}
	
/** 
* Enregistre les nouvelles unités du quadrillage.
*/
	static void newGrid(double[] nval) {
		QX = nval[0] ; QY = nval[1] ; pathGrid() ;
		if (Z != null) Z.repaint() ;
	}
	
/** 
* Normalise le repère.
*/
	static void normal() {
		coords( 
			new double[]{ - ORIGIN.x * UNIT , (W - ORIGIN.x) * UNIT , (ORIGIN.y - H) * UNIT , ORIGIN.y * UNIT} ,
			true
		) ;
	}
	
/*___________________________________________________________________________________________________________/
/   4. TRACÉS.																								 /
/___________________________________________________________________________________________________________*/

// 4.1 contexte

/** 
* Contexte graphique dans lequel le tracé des objets est réalisé.
*/	
    public static Graphics2D GRAPHICS ;

/** 
* FontRenderContext associé à GRAPHICS.
*/	
	static FontRenderContext FRC ;

/** 
* Fonte attribuée à GRAPHICS.
*/	
	public static   Font FONT_PLAIN = new Font("Lucida Sans" , Font.PLAIN , 12) , 

/** 
* Fonte italique dérivée de FONT_PLAIN .
*/		
					FONT_ITALIC = FONT_PLAIN.deriveFont(Font.ITALIC) ;	
	
// 4.2 crayons

/** 
* Crayon par défaut.
*/	
    final static BasicStroke STROKE = new BasicStroke(1.0f) ;

/** 
* Crayon pointillé.
*/	
    final static BasicStroke DASHED = new BasicStroke(
		1.0f, 
		BasicStroke.CAP_BUTT, 
		BasicStroke.JOIN_MITER, 
		2.0f , 
		new float[]{2.0f} , 
		0.0f
	) ;
	
/** 
* Crayon pointillé épais.
*/	
    final static BasicStroke BDASHED = new BasicStroke(
		2f, 
		BasicStroke.CAP_BUTT, 
		BasicStroke.JOIN_MITER, 
		2.0f , 
		new float[]{4.0f , 2.0f} , 
		0.0f
	) ;
	
// 4.2 Couleurs

/** 
* Couleur du trait des points libres.
*/	
	static Color POINT_COLOR = new Color(0 , 0 , 0) ;

/** 
* Couleur du fond des points libres.
*/	
	//static Color POINT_BACK_COLOR = new Color(255 , 208 , 137) ;
	static Color POINT_BACK_COLOR = new Color(255 , 0 , 0) ;

/** 
* Couleur du trait des points d'intersection.
*/	
	static Color POINT_INTER_COLOR = new Color(21 , 42 , 42) ;

/** 
* Couleur du fond des points d'intersection.
*/	
	static Color POINT_INTER_BACK_COLOR = new Color(216 , 216 , 216) ;

/** 
* Couleur du trait des points sur une courbe : celui de la courbe.
*/	

/** 
* Couleur du fond des points sur une courbe.
*/	
	static Color POINT_ON_BACK_COLOR = new Color(216 , 216 , 216) ;

/** 
* Couleur des droites.
*/	
	static Color LINE_COLOR = new Color(21 , 177 , 21) ;

/** 
* Couleur des courbes.
*/	
	static Color CURVE_COLOR = new Color(255 , 21 , 42) ;

/** 
* Couleur des courbes définies par la méthode d'Euler.
*/	
	static Color EULER_COLOR = new Color(21 , 42 , 255) ;

/** 
* Couleur des tangentes.
*/	
	static Color SLOPE_COLOR = new Color(234 , 21 , 234) ;

/** 
* Couleur des équations.
*/	
	static Color EQUATION_COLOR = new Color(188 , 21 , 235) ;

/** 
* Couleur des marques.
*/
	public static Color MARK_COLOR = new Color(105 , 105 , 105) ;

/** 
* Couleur du trait des domaines.
*/	
	static Color DOMAIN_COLOR = new Color(0 , 141 , 177) ;

/** 
* Couleur du fond des domaines.
*/	
	static Color DOMAIN_BACK_COLOR = new Color(157 , 196 , 255 , 128) ;

/** 
* Couleur du trait des suites définies par récurrence.
*/	
	static Color REC_SEQUENCE_COLOR = new Color(128 , 128 , 128) ;

/** 
* Couleur du fond des suites définies par récurrence.
*/	
	static Color REC_SEQUENCE_BACK_COLOR = new Color(128 , 128 , 128) ;

/** 
* Couleur du trait des suites définies par une fonction .
*/	
	static Color GEN_SEQUENCE_COLOR = new Color(255 , 137 , 137) ;

/** 
* Couleur du fond des suites définies par une fonction .
*/	
	static Color GEN_SEQUENCE_BACK_COLOR = new Color(255 , 137 , 137) ;
	
// 4.3 chemins
// 4.3.1 fenêtre

/** 
* Chemin parcourant le bord de la fenêtre-utilisateur. <br>
* Le tracé des objets non bornés comme les droites et les demi-droites utilise ZONE_PATH.
*/
	static GeneralPath ZONE_PATH = new GeneralPath() ;
	
/** 
* Redéfinit ZONE_PATH , c'est-à-dire le chemin parcourant le bord de la fenêtre-utilisateur. 
* Aucun chemin n'est recalculé par cette méthode.
*/
	static void pathWindow() {
		XG = - ORIGIN.x * UNIT / UX ;
		XD = (W - ORIGIN.x) * UNIT / UX ;
		YB = (ORIGIN.y - H) * UNIT / UY ;
		YH = ORIGIN.y * UNIT / UY ;
		ZONE_PATH.reset() ;
		ZONE_PATH.moveTo((float)XG , (float)YH) ;
		ZONE_PATH.lineTo((float)XG , (float)YB) ;
		ZONE_PATH.lineTo((float)XD , (float)YB) ;
		ZONE_PATH.lineTo((float)XD , (float)YH) ;
		ZONE_PATH.closePath() ;
	}

// 4.3.2 axes

/** 
* Chemin parcourant les axes du repère-utilisateur.
*/
	static GeneralPath AXIS_PATH = new GeneralPath() ;

/** 
* Couleur par défaut de AXIS_PATH.
*/
	public static Color AXIS_COLOR = new Color(156 , 156 , 156) ;

/** 
* Redéfinit AXIS_PATH , c'est-à-dire le chemin parcourant les axes du repère-utilisateur. 
*/
	static void pathAxis() {
		AXIS_PATH.reset() ; 
		AXIS_PATH.moveTo(0 , ORIGIN.y) ;
		AXIS_PATH.lineTo(W , ORIGIN.y) ;
		AXIS_PATH.append(arrowPath(new Point2D.Float(W , ORIGIN.y) , 0 , 8) , false) ;
		AXIS_PATH.moveTo(ORIGIN.x , H) ;
		AXIS_PATH.lineTo(ORIGIN.x , 0) ;
		AXIS_PATH.append(arrowPath(new Point2D.Float(ORIGIN.x , 0) , -90 , 8) , false) ;
		Point2D.Float p ;
		p = screenPoint(1 , 0) ; AXIS_PATH.moveTo(p.x , p.y) ;
		movePoint(p , 2 , 90) ; AXIS_PATH.moveTo(p.x , p.y) ;
		movePoint(p , 4 , -90) ; AXIS_PATH.lineTo(p.x , p.y) ;
		p = screenPoint(0 , 1) ; AXIS_PATH.moveTo(p.x , p.y) ;
		movePoint(p , 2 , 180) ; AXIS_PATH.moveTo(p.x , p.y) ;
		movePoint(p , 4 , 0) ; AXIS_PATH.lineTo(p.x , p.y) ;
	}

/** 
* Renvoie un chemin représentant une flèche .  <br>
* Si un chemin atteint ou passe par le point <i> p </i> avec une direction <i> d </i> mesurée en degrés,
* la flèche placée sur ce chemin est construite de la façon suivante :
* <ul>
*   <li> la pointe de la flèche est placée en <i> p </i> </li>
*   <li> les deux extrêmités de la flèche sont à la distance <i> l </i> de <i> p </i> dans les directions <i> d </i> -180 ± 25° </li>
*   <li> les tangentes au tracé en ces extrêmités passent par le point de contrôle situé 
*		 à la distance <i> l/2 </i> de <i> p </i> dans la direction <i> d </i> -180° . </li>
* </ul>
*/	
	static GeneralPath arrowPath(Point2D.Float p , double d , double l) {
		GeneralPath P = new GeneralPath() ;
		if (d > 0) d -= 180 ; else d += 180 ;
		Point2D.Float   a = pointMove(p , l , d - 25) , 
						b = pointMove(p , l/2 , d ) ,
						c = pointMove(p , l , d + 25) ;
		P.moveTo(a.x , a.y) ; 
		P.quadTo(b.x , b.y , p.x , p.y) ; 
		P.quadTo(b.x , b.y , c.x , c.y) ;
		return P ;
	}
	
/** 
* Tableau de points-utilisateur utilisé pour tracer une flèche. 
*/
	static Point2D.Float[] AP = new Point2D.Float[] {
		new Point2D.Float() , new Point2D.Float() , new Point2D.Float() , new Point2D.Float()
	} ;
	
/** 
* Renvoie un chemin-utilisateur représentant un vecteur défini par deux points utilisateurs .  <br>
* Si les points ne sont pas trop éloignés, une flèche est dessinée sur le segment reliant les deux points.
* @param p1 origine
* @param p2 extrêmité
* @param min distance-écran minimale pour que la flèche soit dessinée
* @param pos position du point de contröle
* @param rad rayon-écran de la flèche
* @param deg angle en degrés mesurant l'écartement de la flèche
*/
	static GeneralPath userVect(Point2D.Float p1 , Point2D.Float p2 , float min , float pos , float rad , float deg) {
		GeneralPath P = new GeneralPath() ;
		AP[1].setLocation(p1) ; AP[2].setLocation(p2) ;
		TO_SCREEN.transform(AP[1] , AP[1]) ; TO_SCREEN.transform(AP[2] , AP[2]) ;
		double l = AP[1].distance(AP[2]) ;
		if (l >= min) {
			double d = getDegrees(AP[2] , AP[1]) ;
			AP[0].setLocation(pointMove(AP[2] , (pos != 1) ? l*pos : rad/2 , d)) ;
			if (pos != 1) movePoint(AP[2] , l*pos - rad/2 , d) ;
			AP[1].setLocation(pointMove(AP[2] , rad , d - deg)) ;
			AP[3].setLocation(pointMove(AP[2] , rad , d + deg)) ;
			for(int i = 0 ; i < 4 ; i++) AP[i].setLocation(abs(AP[i].x) , ord(AP[i].y)) ;
			P.moveTo(AP[1].x , AP[1].y) ; 
			P.quadTo(AP[0].x , AP[0].y , AP[2].x , AP[2].y) ;
			P.quadTo(AP[0].x , AP[0].y , AP[3].x , AP[3].y) ;
		}
		P.moveTo(p1.x , p1.y) ; P.lineTo(p2.x , p2.y) ;
		return P ;
	}

/** 
* Renvoie un chemin-écran représentant un vecteur défini par deux points-écran .  <br>
* @param P1 origine
* @param P2 extrêmité
* @param pos position du point de contröle
* @param rad rayon-écran de la flèche
* @param deg angle en degrés mesurant l'écartement de la flèche
*/
	static GeneralPath screenVect(Point2D P1 , Point2D P2 , float pos , float rad , float deg) {
		GeneralPath P = new GeneralPath() ;
		Point2D.Float   p1 = new Point2D.Float ((float)P1.getX() , (float)P1.getY()) ,
						p2 = new Point2D.Float ((float)P2.getX() , (float)P2.getY()) ;
		double l = p1.distance(p2) ;
		double d = getDegrees(p2 , p1) ;
		AP[0].setLocation(pointMove(p2 , (pos != 1) ? l*pos : rad/2 , d)) ;
		AP[2].setLocation(p2) ; if (pos != 1) movePoint(AP[2] , l*pos - rad/2 , d) ;
		AP[1].setLocation(pointMove(AP[2] , rad , d - deg)) ;
		AP[3].setLocation(pointMove(AP[2] , rad , d + deg)) ;
		P.moveTo(AP[1].x , AP[1].y) ; 
		P.quadTo(AP[0].x , AP[0].y , AP[2].x , AP[2].y) ;
		P.quadTo(AP[0].x , AP[0].y , AP[3].x , AP[3].y) ;
		P.moveTo(p1.x , p1.y) ; P.lineTo(p2.x , p2.y) ;
		return P ;
	}

// 4.3.3 quadrillage	
/** 
* Chemin parcourant le quadrillage.
*/
	static GeneralPath GRID_PATH = new GeneralPath() ;

/** 
* Couleur par défaut de GRID_PATH.
*/
	public static Color GRID_COLOR = new Color(210 , 210 , 210) ;

/** 
* Unité du quadrillage sur l'axe des abscisses. 
*/
	static double QX = 1 ;

/** 
* Unité du quadrillage sur l'axe des ordonnées. 
*/
	static double QY = 1 ;

/** 
* Redéfinit GRID_PATH , c'est-à-dire le chemin parcourant le quadrillage . 
*/
	static void pathGrid() {
		GRID_PATH.reset() ; float z ; double m ;
		m = QX ;
		for (double x = m * Math.floor(XG / m) ; x <= XD ; x += m) {
			z = (float)absE(x) ;
			GRID_PATH.moveTo(z , 0) ; GRID_PATH.lineTo(z , H) ;
		}
		m = QY ;
		for (double y = m * Math.floor(YB / m) ; y <= YH ; y += m) {
			z = (float)ordE(y) ;
			GRID_PATH.moveTo(0 , z) ; GRID_PATH.lineTo(W , z) ;
		}
	}
	
// 4.4 Réalisation des tracés

/** 
* Remplit le fond avec la couleur blanche . 
*/
	static void traceFond() {
		GRAPHICS.setColor(Color.white) ; GRAPHICS.fillRect(0 , 0 , W , H) ;
	}

/** 
* Trace le repère . 
*/
	static void traceRepere() {
		GRAPHICS.setColor(AXIS_COLOR) ; GRAPHICS.draw(AXIS_PATH) ;
	}

/** 
* Trace le quadrillage . 
*/
	static void traceQuadrillage() {
		GRAPHICS.setColor(GRID_COLOR) ; GRAPHICS.draw(GRID_PATH) ;
	}


/*___________________________________________________________________________________________________________/
/   5. TESTS.																								 /
/___________________________________________________________________________________________________________*/

// 5.1 tests de définition

/** 
* Teste si un nombre a une valeur réelle finie. 
*/
    static boolean def(double y) { return (!Double.isNaN(y) && !Double.isInfinite(y)) ; }

/** 
* Teste si un nombre a une valeur réelle finie. 
*/
    static boolean def(float y) { return (!Float.isNaN(y) && !Float.isInfinite(y)) ; }

/** 
* Teste si les coordonnées d'un point ont des valeurs réelles finies. 
*/	
	static boolean def(Point2D p) { return (def(p.getX()) && def(p.getY())) ; }

/** 
* Teste si les coordonnées d'un point ont des valeurs réelles finies. 
*/	
	static boolean def(Point2D.Float p) { return (def(p.x) && def(p.y)) ; }

// 5.2 proximité de la souris

/** 
* Valeur en-dessous de laquelle une distance est considérée comme nulle. 
*/
	static double TOLERANCE = 4 ;

/** 
* Teste si un point-écran est proche des axes. 
*/
	static boolean repereProche() {
		return ((Math.abs(ORIGIN.x - MOUSE.x) < TOLERANCE) || (Math.abs(ORIGIN.y - MOUSE.y) < TOLERANCE)) ;
	}
	
/** 
* Teste si la souris est proche d'un point-utilisateur défini par ses coordonnées . 
*/
	static boolean mouseIsNear(float x , float y) {
		return (MOUSE.distance(screenPoint(x , y)) < TOLERANCE) ;
	}
	
/** 
* Teste si la souris est proche d'un point-utilisateur . 
*/
	static boolean mouseIsNear(Point2D p) {
		return (MOUSE.distance(screenPoint(p.getX() , p.getY())) < TOLERANCE) ;
	}
	
/** 
* Calcule la distance entre deux points-écran donnés par leurs coordonnées. 
*/
    static double distE(int A , int B , int C , int D) { 
		return Math.sqrt((C - A) * (C - A) + (D - B) * (D - B)) ;
	}
	
// 5.3 Tests de proximité

/** 
* Teste si deux points-utilisateur sont proches . 
*/
	static boolean isNear(Point2D p , Point2D q) {
		return (screenPoint(p.getX() , p.getY()).distance(screenPoint(q.getX() , q.getY())) < TOLERANCE) ;
	}

/** 
* Teste si deux points-utilisateur sont indiscernables . 
*/
	static boolean isVisualEqu(Point2D p , Point2D q) {
		Point2D.Float a = screenPoint(p.getX() , p.getY()) , b = screenPoint(q.getX() , q.getY()) ;
		return ((Math.rint(b.x -a.x) == 0) && (Math.rint(b.y -a.y) == 0)) ;
	}

/** 
* Teste si un point-utilisateur est visible à l'écran  . 
*/
	static boolean isVisible(Point2D p) {
		if (!def(p)) return false ;
		int XE = absE(p.getX()) , YE = ordE(p.getY()) ;
		return ((0 <= XE) && (XE <= W) && (0 <= YE) && (YE <= H)) ; 
	}

/*___________________________________________________________________________________________________________/
/   6. MISE EN FORME																						 /
/___________________________________________________________________________________________________________*/

// 6.1 Arrondis

/** 
* L'objet <i>Locale</i> utilisé par l'application.<br>
* Pour utiliser un objet intrenational, remplacer par :
*   public static Locale LOCALE = new Locale("en" , "US") ;
*/
	public static Locale LOCALE = Locale.getDefault() ;
	
/** 
* Formatage de données numériques. 
*/
	public static DecimalFormat NF = (DecimalFormat)NumberFormat.getInstance(LOCALE) ;
	
/** 
* Séparateur décimal. 
*/
	public static char DSEP = NF.getDecimalFormatSymbols().getDecimalSeparator() ;

/** 
* Séparateur d'informations. 
*/
	public static char ISEP = (DSEP == ',') ? ';' : ',' ;
	
	
/** 
* Arrondi pour l'affichage de résultats numériques. 
*/
	static double ROUND = .01 ;
	
/** 
* Nombre de chiffres après la virgule affichés pour un résultat numérique. 
*/
	static int DIGITS = 2 ;
	
/** 
* Mise en forme d'un résultat numérique. <br>
* Le résultat est arrondi de façon que le nombre de chiffres après la virgule soit au plus <i> di </i> .
*/
	static String form(double x , int di) {
		NF.setMaximumFractionDigits(di) ;
		String s = NF.format(x) ;
		NF.setMaximumFractionDigits(DIGITS) ;
		if (s.equals("-0")) return "0" ; else return s ;
	}
    
/** 
* Mise en forme d'un résultat numérique avec un nombre de chiffres après la virgule qui est au plus DIGITS . <br>
*/
	static String form(double x) { 
		String s = NF.format(x) ; if (s.equals("-0")) return "0" ; else return s ;
	}
	
/** 
* Mise en forme des coordonnées d'un point avec un nombre de chiffres après la virgule qui est au plus DIGITS .
*/
	static String form(Point2D p) { return "(" + form(p.getX()) + " " + ISEP + " " + form(p.getY()) + ")" ; }
	
	static String form(String s) {
		if (DSEP == '.') return s ;
		StringBuffer r = new StringBuffer(s) ; int n ;
		while(true) {
			n = r.indexOf(".") ; if (n < 0) break ;
			r.setCharAt(n , DSEP) ;
		}
		return new String(r) ;
	}

/*___________________________________________________________________________________________________________/
/   7. INTERPRÉTATION																 /
/___________________________________________________________________________________________________________*/

//  7.1 chaînes numériques

/** 
* Retourne la valeur booléenne d'une chaîne.
*/
	static boolean boolv(String s) { return Boolean.valueOf(s).booleanValue() ; }
	
/** 
* Retourne la valeur entière d'une chaîne.
*/
	static int intv(String s) {
		try { return Integer.parseInt(s) ; }
		catch (Exception exc) { return Integer.MAX_VALUE ; }
	}
	
/** 
* Retourne un <i> float </i> qui représente la valeur réelle d'une chaîne.
*/
	static float fltv(String s) {
		try { return Float.parseFloat(s) ; }
		catch (Exception exc) {
			try { return (NF.parse(s)).floatValue() ;  }
			catch (ParseException pexc) { return Float.NaN ; }
		}
	}
	
/** 
* Retourne un <i> double </i> qui représente la valeur réelle d'une chaîne.
*/
	static double dblv(String s) {
		try { return Double.parseDouble(s) ; }
		catch (Exception exc) { 
			try { return (NF.parse(s)).doubleValue() ;  }
			catch (ParseException pexc) {return Double.NaN ; }
		}
	}
	
//  7.2 Coordonnées

/** 
* Analyse une chaîne représentant les coordonnées d'un point.
*/
	static void putValues(Point2D.Float p , String text) {
		int i = 0 ;
		form f ; 
		nbre n = null ;
		StringTokenizer st = new StringTokenizer(text , "\u0028\u0029" + ISEP) ;
		while(st.hasMoreTokens() && (i < 2)) {
			f = calcul.toForm(st.nextToken()) ;
			n = (f != null) ? f.eval(calcul.EVAL_POLYS).getNbre() : null ;
			if (n != null) {
				if (i == 0) p.x = (float)n.val ; else p.y = (float)n.val ;
			}
			i ++ ;
		}
	}

//		7.3 Formules.

/** 
* Interprète une chaîne comme une formule. Renvoie la valeur numérique associée.
*/
	static double getValue(String text) {
		form f = calcul.toForm(text) ;
		if (f != null) {
			nbre n = f.eval(calcul.EVAL_POLYS).getNbre() ;
			if (n != null) return n.val ;
		}
		return Double.NaN ;
	}


/*___________________________________________________________________________________________________________/
/   8. INTERSECTIONS																									 /
/___________________________________________________________________________________________________________*/

// 8.1 Intersection de droites
/** 
* Renvoie le produit scalaire des vecteurs <i> (a , b) </i> et <i> (a , c) </i>  .
*/	
	static float dotProduct(Point2D.Float a , Point2D.Float b , Point2D.Float c) {
		return (b.x - a.x) * (c.x - a.x) + (b.y - a.y) * (c.y - a.y) ;
	}

/** 
* Renvoie le point d'intersection des droites <i> (a , b) </i> et <i> (c , d) </i>  .  <br>
* Si les droites sont parallèles , les coordonnées sont (Float.NaN , Float.NaN) .
*/	
	static Point2D.Float pointInter(Point2D.Float a , Point2D.Float b , Point2D.Float c , Point2D.Float d) {
		float e = (b.x - a.x) * (d.y - c.y) - (b.y - a.y) * (d.x - c.x) ;
		if ((e == 0) || Float.isNaN(e)) return new Point2D.Float(Float.NaN , Float.NaN) ;
		float f = ((c.x - a.x) * (d.y - c.y) - (c.y - a.y) * (d.x - c.x)) / e ;
		return new Point2D.Float(a.x + f * (b.x - a.x) , a.y + f * (b.y - a.y)) ;
	}
	
/** 
* Met à jour le point d'intersection des droites <i> (a , b) </i> et <i> (c , d) </i>  .  <br>
* Si les droites sont parallèles , les coordonnées sont (Float.NaN , Float.NaN) .
*/	
	static void setPointInter(Point2D.Float p , Point2D.Float a , Point2D.Float b , Point2D.Float c , Point2D.Float d) {
		float e = (b.x - a.x) * (d.y - c.y) - (b.y - a.y) * (d.x - c.x) ;
		if ((e == 0) || Float.isNaN(e)) p.setLocation(Float.NaN , Float.NaN) ;
		else {
			float f = ((c.x - a.x) * (d.y - c.y) - (c.y - a.y) * (d.x - c.x)) / e ;
			p.setLocation(a.x + f * (b.x - a.x) , a.y + f * (b.y - a.y)) ;
		}
	}
	
// 8.2 Intersection de courbes

/** 
* Intersection de deux courbes par dichotomie.<br>
* On résout f(x) = g(x) sur l'intervalle [u , v] . Si (f - g)(u) et (f - g)(v) sont de même signe , 
* le résultat est Float.NaN.<br>
* Sinon, on teste x = (u + v)/2 : 
* <ul type="circle">
* <li> si (f - g)(u) et (f - g)(x) sont de signes différents, on réitère le processus sur [u , x] </li>
* <li> si (f - g)(x) et (f - g)(v) sont de signes différents, on réitère le processus sur [x , v] </li>
* <li> dans les autres cas, le processus est arrêté </li>
* </ul>
* Le processus est poursuivi jusqu'à ce que les deux abscisses ne soient plus discernables à l'écran.
* @param e la première courbe.
* @param f la deuxième courbe.
* @param u borne gauche de l'intervalle.
* @param v borne droite de l'intervalle.
*/
	static float solveD(line e , line f , float u , float v) {
		float   x1 = u ,
				x2 = v ,
				x ,
				y1 = (float)(e.computeValue(x1) - f.computeValue(x1)) ,
				y2 = (float)(e.computeValue(x2) - f.computeValue(x2)) ,
				y ; 
		if (!def(x1) || !def(x2) || !def(y1) || !def(y2) || (y1 * y2 > 0)) return Float.NaN ;
		else if (y1 == 0) return x1 ;
		else if (y2 == 0) return x2 ;
		double epx = Math.min(UNIT / UX , (v - u)/8) ;
		while (x2 - x1 > epx) {
			x = (x1 + x2) / 2 ; y = (float)(e.computeValue(x) - f.computeValue(x)) ;
			if (y == 0) return x ;
			else if (!def(y)) return Float.NaN ;
			else if (y1 * y < 0) { x2 = x ; y2 = y ; }
			else { x1 = x ; y1 = y ; }
		}
		return (x1*y2 - x2*y1) / (y2 - y1) ;
		
	}

/** 
* Intersection de deux courbes par approximation du second degré.<br>
* On résout f(x) = g(x) sur l'intervalle [u , v] où (f - g)(u) et (f - g)(v) sont de même signe et proches de 0 <br>
* Pour cela, on approche la courbe de  f - g  par un arc de parabole en calculant ses valeurs en <i>u</i> , en en <i>v</i>
* et en (en <i>u</i> + en <i>v</i>) / 2 . On détermine ensuite l'abscisse du sommet : c'est le résultat 
* si elle est entre les bornes et sinon, on renvoie Float.NaN. 
* @param e la première courbe.
* @param f la deuxième courbe.
* @param u borne gauche de l'intervalle.
* @param v borne droite de l'intervalle.
*/
	static float solveT(line e , line f , float u , float v) {
		float   x1 = u ,
				x2 = v ,
				x0 = (u + v) / 2 ,
				x = Float.NaN ,
				y1 = (float)(e.computeValue(x1) - f.computeValue(x1)) ,
				y2 = (float)(e.computeValue(x2) - f.computeValue(x2)) ,
				y0 = (float)(e.computeValue(x0) - f.computeValue(x0)) ,
				y ; 
		if (!def(y1) || !def(y2) || !def(y0)) {
			return Float.NaN ;
		}
		if (y1 == 0) x = x1 ; else if (y2 == 0) x = x2 ; else if (y0 == 0) x = x0 ;
		if (def(x)) {
			return x ;
		}
		x = x0 + (y2 - y1) * (x2 - x1) / (2*y0 - y1 - y2) / 4 ;
		y = (float)(e.computeValue(x) - f.computeValue(x)) ;
		if (((x >= x1) && (x <= x2)) || (y == 0)) return x ; else return Float.NaN ;
	}

// 8.3 Équations
	
/** 
* Résolution d'une équation par dichotomie.<br>
* On résout f(x) = 0 sur l'intervalle [u , v] . Si f(u) et f(v) sont de même signe , le résultat est Float.NaN.<br>
* Le processus est poursuivi jusqu'à ce que les deux abscisses ne soient plus discernables à l'écran.
* @param e l'objet <i>form</i> représentant la fonction.
* @param sv la variable.
* @param u borne gauche de l'intervalle.
* @param v borne droite de l'intervalle.
*/
	static float solveD(form e , String sv , float u , float v) {
		float   x1 = u ,
				x2 = v ,
				x ,
				y1 = (float)e.toDouble(sv , x1) ,
				y2 = (float)e.toDouble(sv , x2) ,
				y ; 
		if (!def(x1) || !def(x2) || !def(y1) || !def(y2) || (y1 * y2 > 0)) return Float.NaN ;
		else if (y1 == 0) return x1 ;
		else if (y2 == 0) return x2 ;
		double epx = Math.min(UNIT / UX , (v - u)/8) ;
		while (x2 - x1 > epx) {
			x = (x1 + x2) / 2 ; y = (float)e.toDouble(sv , x) ;
			if (y == 0) return x ;
			else if (!def(y)) return Float.NaN ;
			else if (y1 * y < 0) { x2 = x ; y2 = y ; }
			else { x1 = x ; y1 = y ; }
		}
		return (x1*y2 - x2*y1) / (y2 - y1) ;
		
	}

/*___________________________________________________________________________________________________________/
/   9. ZONE																									 /
/___________________________________________________________________________________________________________*/

/** 
* L'objet <i> zone </i> qui est associé à cette classe.
*/
	public static zone Z ;

/** 
* L'objet <i> PrefsPane </i> qui est associé à cette classe.
*/
	public static PrefsPane P ;

/** 
* L'objet <i> dial </i> qui est associé à cette classe.
*/
	public static dial D ;

/** 
* Instructions par défaut.
*/
    public static void setDefault(zone z) {
		StringBuffer sb = new StringBuffer("barre(point") ;
		sb.append(',') ;  sb.append("pointoncurve") ;
		sb.append(',') ;  sb.append("intersection") ;
		sb.append(',') ;  sb.append("mark") ;
		sb.append(',') ;  sb.append("space") ;
		sb.append(',') ;  sb.append("line") ;
		sb.append(',') ;  sb.append("tangent") ;
		sb.append(',') ;  sb.append("curve") ;
		sb.append(',') ;  sb.append("euler") ;
		sb.append(',') ;  sb.append("space") ;
		sb.append(',') ;  sb.append("recsequence") ;
		sb.append(',') ;  sb.append("gensequence") ;
		sb.append(',') ;  sb.append("equation") ;
		sb.append(',') ;  sb.append("space") ;
		sb.append(',') ;  sb.append("domain") ;
		sb.append(',') ;  sb.append("space") ;
		sb.append(',') ;  sb.append("select") ;
		sb.append(',') ;  sb.append("delete") ;
		sb.append(',') ;  sb.append("zoom") ;
		sb.append(',') ;  sb.append("zoomp") ;
		sb.append(',') ;  sb.append("zoomm") ;
		sb.append(',') ;  sb.append("tab") ;
		sb.append(',') ;  sb.append("aspect") ;
		sb.append(',') ;  sb.append("prefs") ;
		sb.append(',') ;  sb.append("help)") ;
		z.dconfiguration = new String(sb) ;
		sb = new StringBuffer("fenetre(") ; sb.append(600) ;
		sb.append(',') ;  sb.append(400) ;
		sb.append(',') ;  sb.append(-6.5) ; sb.append(',') ;  sb.append(6.5) ;
		sb.append(',') ;  sb.append(-3.5) ; sb.append(',') ;  sb.append(5.5) ;
		sb.append(',') ;  sb.append(.1) ; sb.append(',') ;  sb.append(.1) ;
		sb.append(',') ;  sb.append(1) ; sb.append(',') ;  sb.append(1) ;
		sb.append(',') ;  sb.append(.01) ;
		sb.append(',') ;  sb.append(true) ; sb.append(',') ;  sb.append(false) ;
		sb.append(',') ;  sb.append("Lucida Sans") ; sb.append(',') ;  sb.append(12) ;
		sb.append(',') ;  sb.append(1) ; sb.append(',') ;  sb.append(1) ;
		sb.append(',') ;  sb.append(156) ; sb.append(',') ;  sb.append(156) ; sb.append(',') ;  sb.append(156) ;
		sb.append(',') ;  sb.append(210) ; sb.append(',') ;  sb.append(210) ; sb.append(',') ;  sb.append(210) ;
		sb.append(')') ;
		z.dfenetre = new String(sb) ;
	}

/** 
* Affecte un objet <i> zone </i> à cette classe.
*/
    public static void setZone(zone z) {
		Z = z ;
		setGraphics(Z.getGraphics()) ; 
		setSize(Z.getWidth() , Z.getHeight()) ;
	}

/** 
* Initialise le contexte graphique de cette classe.
*/
    public static void setGraphics(Graphics gg) {
        GRAPHICS = (Graphics2D) gg ;
        GRAPHICS.setRenderingHint(RenderingHints.KEY_ANTIALIASING , RenderingHints.VALUE_ANTIALIAS_ON) ;
		FRC = GRAPHICS.getFontRenderContext() ;
		GRAPHICS.setFont(FONT_PLAIN) ;
		box.setLine() ;
	}

/** 
* Modifie les dimensions de la fenêtre et recalcule les chemins si nécessaire.
*/
    public static void setSize(int w , int h) {
		if ((W == w) && (H == h)) return ;
		W = w ; H = h ;
		pathWindow() ; 
		if (Z.quadrillageVisible) pathGrid() ; 
		if (Z.repereVisible) pathAxis() ;
		setShapes() ;
	}

/** 
* Modifie la fonte par défaut.
*/
    public static void setFont(Font f) { 
		FONT_PLAIN = f ;
		FONT_ITALIC = FONT_PLAIN.deriveFont(Font.ITALIC) ;
		if (GRAPHICS != null) GRAPHICS.setFont(FONT_PLAIN) ;
		box.setLine() ;
	}

/** 
* Déplace l'origine du repère-utilisateur lorsque tous les paramètres ont été calculés.
*/
	static void setOrigin() { setOrigin(W * XG / (XG - XD) , H * YH / (YH - YB)) ; }
	
/** 
* Déplace l'origine du repère-utilisateur.
* Met à jour TO_SCREEN , ZONE_PATH , GRID_PATH et AXIS_PATH .
* La fenêtre des préférences reçoit les nouvelles données. La barre de dialogue est ensuite remise à jour.
* Tous les chemins représentant les objets graphiques sont recalculés.
*/
	static void setOrigin(double x , double y) {
		ORIGIN.setLocation(x , y) ; 
		TO_SCREEN.setTransform(UX / UNIT , 0 , 0 , -UY / UNIT , ORIGIN.x , ORIGIN.y) ;
		pathWindow() ;
		if (Z.quadrillageVisible) pathGrid() ;
		if (Z.repereVisible) pathAxis() ;
		P.refresh() ;
		D.refresh() ;
		setShapes() ;
		Z.repaint() ;
	}
	
/** 
* Cette méthode est appelée à la fin d'un déplacement.
* Les chemins des objets sont recalculés avec les nouvelles données. 
*/
	static void setShapes() {
		objetG o ;
		for(int i = 0 ; i < Z.GR.size() ; i++) {
			o = (objetG)Z.GR.elementAt(i) ; o.calcShape() ;
		}
	}
	
/*___________________________________________________________________________________________________________/
/   10. ENTRÉES/SORTIES																								 /
/___________________________________________________________________________________________________________*/


/** 
* Chaîne où sont enregistrées les données actuelles de la classe.
*/
	public static String getDatas() {
		StringBuffer sb = new StringBuffer("fenetre(") ; sb.append(W) ;
		sb.append(',') ; sb.append(H) ;
		sb.append(',') ; sb.append(XG) ; sb.append(',') ; sb.append(XD) ;
		sb.append(',') ; sb.append(YB) ; sb.append(',') ; sb.append(YH) ;
		sb.append(',') ; sb.append(USER_GRID) ; sb.append(',') ; sb.append(USER_GRID) ;
		sb.append(',') ; sb.append(QX) ; sb.append(',') ; sb.append(QY) ;
		sb.append(',') ; sb.append(ROUND) ;
		sb.append(',') ; sb.append(Z.repereVisible) ; sb.append(',') ; sb.append(Z.quadrillageVisible) ;
		sb.append(',') ; sb.append(P.comboFont.getSelectedItem()) ; 
		sb.append(',') ; sb.append(P.comboSize.getSelectedItem()) ;
		sb.append(',') ; sb.append(UX) ; sb.append(',') ; sb.append(UY) ;
		sb.append(',') ; sb.append(AXIS_COLOR.getRed()) ;
		sb.append(',') ; sb.append(AXIS_COLOR.getGreen()) ;
		sb.append(',') ; sb.append(AXIS_COLOR.getBlue()) ;
		sb.append(',') ; sb.append(GRID_COLOR.getRed()) ;
		sb.append(',') ; sb.append(GRID_COLOR.getGreen()) ;
		sb.append(',') ; sb.append(GRID_COLOR.getBlue()) ;
		sb.append(')') ;
		return new String(sb) ;
	}

/** 
* Mise à jour des données actuelles de la classe.
*/
	public static void setDatas(String[] A) {
		int n = -1 , r , g , b ;
		n++ ; W = intv(A[n]) ;
		n++ ; H = intv(A[n]) ;
		n++ ; XG = dblv(A[n]) ;
		n++ ; XD = dblv(A[n]) ;
		n++ ; YB = dblv(A[n]) ;
		n++ ; YH = dblv(A[n]) ;
		n++ ; USER_GRID = dblv(A[n]) ;
		n++ ;
		n++ ; QX = dblv(A[n]) ;
		n++ ; QY = dblv(A[n]) ;
		n++ ; ROUND = dblv(A[n]) ;
		n++ ; Z.repereVisible = boolv(A[n]) ; 
		n++ ; Z.quadrillageVisible = boolv(A[n]) ;
		n++ ; P.comboFont.setSelectedItem(A[n]) ; 
		n++ ; P.comboSize.setSelectedItem(A[n]) ;
		if (A.length > 15) { //nouveau format
			n++ ; UX = dblv(A[n]) ;  
			n++ ; UY = dblv(A[n]) ;  
			n++ ; r = intv(A[n]) ; n++ ; g = intv(A[n]) ; n++ ; b = intv(A[n]) ; AXIS_COLOR  = new Color(r , g , b) ;
			n++ ; r = intv(A[n]) ; n++ ; g = intv(A[n]) ; n++ ; b = intv(A[n]) ; GRID_COLOR  = new Color(r , g , b) ;
		}
		P.checkRep.setSelected(Z.repereVisible) ; P.checkQuad.setSelected(Z.quadrillageVisible) ;
		if (Z.outOfApplet) {
			Z.G.checkRep.setSelected(Z.repereVisible) ; 
			Z.G.checkQuad.setSelected(Z.quadrillageVisible) ;
		}
		DIGITS = (ROUND < 1) ? -(int)Math.rint(Math.log(ROUND) / Math.log(10)) : (int) ROUND ;
		NF.setMaximumFractionDigits(DIGITS) ;
		P.refresh() ;
		Z.setPreferredSize(new Dimension(W , H)) ;
		UX = W * UNIT /(XD - XG) ; UY = H * UNIT /(YH - YB) ;
		setOrigin() ;
	}

/** 
* L'objet <i> RessourceBundle </i> qui est associé à cette classe.
*/
	public static ResourceBundle RB = ResourceBundle.getBundle ("appstrings", LOCALE) ;

/** 
* L'objet <i> Vector </i> qui est associé à cette classe.
* Utilisé pour extraire des données de RB.
*/
	static Vector V = new Vector() ;
	
	
/** 
* Extraction de données.
* Utilisé pour extraire des données de RB. Renvoie un tableau de chaîne.<br>
* @param s la clé à laquelle est associée une chaîne dans <i>appstrings</i> <br>
* La virgule est le séparateur qui sépare les éléments de de la chaîne associée à la clé.
*/
	public static String[] extract(String s) {
		V.clear() ; int i = 0 , j ; String r = RB.getString(s) ;
		do {
			j = r.indexOf(',' , i) ;
			if (j < 0) { V.add(r.substring(i)) ; break ; }
			else { V.add(r.substring(i , j)) ; i = j + 1 ; }
		}
		while(i < r.length()) ;
		String [] R = new String[V.size()] ; V.toArray(R) ;
		return R ;
	}
	
/** 
* Tableaux de chaînes utilisés par les dialogues.
*/
	static String[]
		strline = extract("strline") ,
		strslope = extract("strslope") ,
		strpoint = extract("strpoint") ,
		strpointon = extract("strpointon") ,
		strsection = extract("strsection") ,
		strrec = extract("strrec") ,
		strequ = extract("strequ") ,
		strdom = extract("strdom") ,
		strcurve = extract("strcurve") ,
		strray = extract("strray") ,
		streuler = extract("streuler") ,
		strgen = extract("strgen") ,
		strpointinter = extract("strpointinter") ,
		strmark = extract("strmark") ,
		strtext = extract("strtext") ,
		strbox = extract("strbox") ,
		strsymb = extract("strsymb")
	;

/** 
* Nombre indiquant que la création d'une courbe est demandée.
*/
public final static int CREATE_CURVE = 1 ;

/** 
* Nombre indiquant que la création d'un point sur une courbe est demandée.
*/
public final static int CREATE_POINT_ON_CURVE = 2 ;

/** 
* Nombre indiquant que la création d'une suite définie par récurrence est demandée.
*/
public final static int CREATE_REC_SEQUENCE = 3 ;

/** 
* Nombre indiquant que la création d'une suite définie par une fonction est demandée.
*/
public final static int CREATE_GEN_SEQUENCE = 4 ;

/** 
* Nombre indiquant que la création d'une courbe solution d'une équation différentielle est demandée.
*/
public final static int CREATE_EULER = 5 ;

/** 
* Nombre indiquant que la création d'un domaine est demandée.
*/
public final static int CREATE_DOMAIN = 6 ;

/** 
* Nombre indiquant que la création d'un point libre est demandée.
*/
public final static int CREATE_POINT = 7 ;

/** 
* Nombre indiquant que la création d'une droite est demandée.
*/
public final static int CREATE_LINE = 8 ;

/** 
* Nombre indiquant que la création d'une tangente est demandée.
*/
public final static int CREATE_TANGENT = 9 ;

/** 
* Nombre indiquant que la création d'un point d'intersection est demandée.
*/
public final static int CREATE_POINT_INTER = 10 ;

/** 
* Nombre indiquant que la création d'une équation est demandée.
*/
public final static int CREATE_EQUATION = 11 ;

/** 
* Nombre indiquant que la création d'une marque est demandée.
*/
public final static int CREATE_MARK = 12 ;


/** 
* Nombre indiquant que la création d'un texte est demandée.
*/
public final static int CREATE_TEXT = 13 ;

}

